"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LegacyCoreEditor = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _brace = _interopRequireDefault(require("brace"));
var _jquery = _interopRequireDefault(require("jquery"));
var _ace_token_provider = require("../../../lib/ace_token_provider");
var curl = _interopRequireWildcard(require("../sense_editor/curl"));
var _smart_resize = _interopRequireDefault(require("./smart_resize"));
var InputMode = _interopRequireWildcard(require("./mode/input"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const _AceRange = _brace.default.acequire('ace/range').Range;
const rangeToAceRange = ({
  start,
  end
}) => new _AceRange(start.lineNumber - 1, start.column - 1, end.lineNumber - 1, end.column - 1);
class LegacyCoreEditor {
  constructor(editor, actions) {
    (0, _defineProperty2.default)(this, "_aceOnPaste", void 0);
    (0, _defineProperty2.default)(this, "$actions", void 0);
    (0, _defineProperty2.default)(this, "resize", void 0);
    (0, _defineProperty2.default)(this, "setActionsBar", (value, topOrBottom = 'top') => {
      if (value === null) {
        this.$actions.css('visibility', 'hidden');
      } else {
        if (topOrBottom === 'top') {
          this.$actions.css({
            bottom: 'auto',
            top: value,
            visibility: 'visible'
          });
        } else {
          this.$actions.css({
            top: 'auto',
            bottom: value,
            visibility: 'visible'
          });
        }
      }
    });
    (0, _defineProperty2.default)(this, "hideActionsBar", () => {
      this.setActionsBar(null);
    });
    this.editor = editor;
    this.$actions = (0, _jquery.default)(actions);
    this.editor.setShowPrintMargin(false);
    const session = this.editor.getSession();
    // @ts-expect-error
    // ignore ts error here due to type definition mistake in brace for setMode(mode: string): void;
    // this method accepts string or SyntaxMode which is an object. See https://github.com/ajaxorg/ace/blob/13dc911dbc0ea31ca343d5744b3f472767458fc3/ace.d.ts#L467
    session.setMode(new InputMode.Mode());
    session.setFoldStyle('markbeginend');
    session.setTabSize(2);
    session.setUseWrapMode(true);
    this.resize = (0, _smart_resize.default)(this.editor);

    // Intercept ace on paste handler.
    this._aceOnPaste = this.editor.onPaste;
    this.editor.onPaste = this.DO_NOT_USE_onPaste.bind(this);
    this.editor.setOptions({
      enableBasicAutocompletion: true
    });
    this.editor.$blockScrolling = Infinity;
    this.hideActionsBar();
    this.editor.focus();
  }

  // dirty check for tokenizer state, uses a lot less cycles
  // than listening for tokenizerUpdate
  waitForLatestTokens() {
    return new Promise(resolve => {
      const session = this.editor.getSession();
      const checkInterval = 25;
      const check = () => {
        // If the bgTokenizer doesn't exist, we can assume that the underlying editor has been
        // torn down, e.g. by closing the History tab, and we don't need to do anything further.
        if (session.bgTokenizer) {
          // Wait until the bgTokenizer is done running before executing the callback.
          if (session.bgTokenizer.running) {
            setTimeout(check, checkInterval);
          } else {
            resolve();
          }
        }
      };
      setTimeout(check, 0);
    });
  }
  getLineState(lineNumber) {
    const session = this.editor.getSession();
    return session.getState(lineNumber - 1);
  }
  getValueInRange(range) {
    return this.editor.getSession().getTextRange(rangeToAceRange(range));
  }
  getTokenProvider() {
    return new _ace_token_provider.AceTokensProvider(this.editor.getSession());
  }
  getValue() {
    return this.editor.getValue();
  }
  async setValue(text, forceRetokenize) {
    const session = this.editor.getSession();
    session.setValue(text);
    if (forceRetokenize) {
      await this.forceRetokenize();
    }
  }
  getLineValue(lineNumber) {
    const session = this.editor.getSession();
    return session.getLine(lineNumber - 1);
  }
  getCurrentPosition() {
    const cursorPosition = this.editor.getCursorPosition();
    return {
      lineNumber: cursorPosition.row + 1,
      column: cursorPosition.column + 1
    };
  }
  clearSelection() {
    this.editor.clearSelection();
  }
  getTokenAt(pos) {
    const provider = this.getTokenProvider();
    return provider.getTokenAt(pos);
  }
  insert(valueOrPos, value) {
    if (typeof valueOrPos === 'string') {
      this.editor.insert(valueOrPos);
      return;
    }
    const document = this.editor.getSession().getDocument();
    document.insert({
      column: valueOrPos.column - 1,
      row: valueOrPos.lineNumber - 1
    }, value || '');
  }
  moveCursorToPosition(pos) {
    this.editor.moveCursorToPosition({
      row: pos.lineNumber - 1,
      column: pos.column - 1
    });
  }
  replace(range, value) {
    const session = this.editor.getSession();
    session.replace(rangeToAceRange(range), value);
  }
  getLines(startLine, endLine) {
    const session = this.editor.getSession();
    return session.getLines(startLine - 1, endLine - 1);
  }
  replaceRange(range, value) {
    const pos = this.editor.getCursorPosition();
    this.editor.getSession().replace(rangeToAceRange(range), value);
    const maxRow = Math.max(range.start.lineNumber - 1 + value.split('\n').length - 1, 1);
    pos.row = Math.min(pos.row, maxRow);
    this.editor.moveCursorToPosition(pos);
    // ACE UPGRADE - check if needed - at the moment the above may trigger a selection.
    this.editor.clearSelection();
  }
  getSelectionRange() {
    const result = this.editor.getSelectionRange();
    return {
      start: {
        lineNumber: result.start.row + 1,
        column: result.start.column + 1
      },
      end: {
        lineNumber: result.end.row + 1,
        column: result.end.column + 1
      }
    };
  }
  getLineCount() {
    // Only use this function to return line count as it uses
    // a cache.
    return this.editor.getSession().getLength();
  }
  addMarker(range) {
    return this.editor.getSession().addMarker(rangeToAceRange(range), 'ace_snippet-marker', 'fullLine', false);
  }
  removeMarker(ref) {
    this.editor.getSession().removeMarker(ref);
  }
  getWrapLimit() {
    return this.editor.getSession().getWrapLimit();
  }
  on(event, listener) {
    if (event === 'changeCursor') {
      this.editor.getSession().selection.on(event, listener);
    } else if (event === 'changeSelection') {
      this.editor.on(event, listener);
    } else {
      this.editor.getSession().on(event, listener);
    }
  }
  off(event, listener) {
    if (event === 'changeSelection') {
      this.editor.off(event, listener);
    }
  }
  isCompleterActive() {
    return Boolean(this.editor.completer && this.editor.completer.activated);
  }
  detachCompleter() {
    // In some situations we need to detach the autocomplete suggestions element manually,
    // such as when navigating away from Console when the suggestions list is open.
    const completer = this.editor.completer;
    return completer === null || completer === void 0 ? void 0 : completer.detach();
  }
  forceRetokenize() {
    const session = this.editor.getSession();
    return new Promise(resolve => {
      // force update of tokens, but not on this thread to allow for ace rendering.
      setTimeout(function () {
        let i;
        for (i = 0; i < session.getLength(); i++) {
          session.getTokens(i);
        }
        resolve();
      });
    });
  }

  // eslint-disable-next-line @typescript-eslint/naming-convention
  DO_NOT_USE_onPaste(text) {
    if (text && curl.detectCURL(text)) {
      const curlInput = curl.parseCURL(text);
      this.editor.insert(curlInput);
      return;
    }
    this._aceOnPaste.call(this.editor, text);
  }
  execCommand(cmd) {
    this.editor.execCommand(cmd);
  }
  getContainer() {
    return this.editor.container;
  }
  setStyles(styles) {
    this.editor.getSession().setUseWrapMode(styles.wrapLines);
    this.editor.container.style.fontSize = styles.fontSize;
  }
  registerKeyboardShortcut(opts) {
    this.editor.commands.addCommand({
      exec: opts.fn,
      name: opts.name,
      bindKey: opts.keys
    });
  }
  unregisterKeyboardShortcut(command) {
    // @ts-ignore
    this.editor.commands.removeCommand(command);
  }
  legacyUpdateUI(range) {
    if (!this.$actions) {
      return;
    }
    if (range) {
      // elements are positioned relative to the editor's container
      // pageY is relative to page, so subtract the offset
      // from pageY to get the new top value
      const offsetFromPage = (0, _jquery.default)(this.editor.container).offset().top;
      const startLine = range.start.lineNumber;
      const startColumn = range.start.column;
      const firstLine = this.getLineValue(startLine);
      const maxLineLength = this.getWrapLimit() - 5;
      const isWrapping = firstLine.length > maxLineLength;
      const totalOffset = offsetFromPage - (window.pageYOffset || 0);
      const getScreenCoords = line => this.editor.renderer.textToScreenCoordinates(line - 1, startColumn).pageY - totalOffset;
      const topOfReq = getScreenCoords(startLine);
      if (topOfReq >= 0) {
        const {
          bottom: maxBottom
        } = this.editor.container.getBoundingClientRect();
        if (topOfReq > maxBottom - totalOffset) {
          this.setActionsBar(0, 'bottom');
          return;
        }
        let offset = 0;
        if (isWrapping) {
          // Try get the line height of the text area in pixels.
          const textArea = (0, _jquery.default)(this.editor.container.querySelector('textArea'));
          const hasRoomOnNextLine = this.getLineValue(startLine).length < maxLineLength;
          if (textArea && hasRoomOnNextLine) {
            // Line height + the number of wraps we have on a line.
            offset += this.getLineValue(startLine).length * textArea.height();
          } else {
            if (startLine > 1) {
              this.setActionsBar(getScreenCoords(startLine - 1));
              return;
            }
            this.setActionsBar(getScreenCoords(startLine + 1));
            return;
          }
        }
        this.setActionsBar(topOfReq + offset);
        return;
      }
      const bottomOfReq = this.editor.renderer.textToScreenCoordinates(range.end.lineNumber, range.end.column).pageY - offsetFromPage;
      if (bottomOfReq >= 0) {
        this.setActionsBar(0);
        return;
      }
    }
  }
  registerAutocompleter(autocompleter) {
    // Hook into Ace

    // disable standard context based autocompletion.
    // @ts-ignore
    _brace.default.define('ace/autocomplete/text_completer', ['require', 'exports', 'module'], function (require, exports) {
      exports.getCompletions = function (innerEditor, session, pos, prefix, callback) {
        callback(null, []);
      };
    });
    const langTools = _brace.default.acequire('ace/ext/language_tools');
    langTools.setCompleters([{
      identifierRegexps: [/[a-zA-Z_0-9\.\$\-\u00A2-\uFFFF]/ // adds support for dot character
      ],
      getCompletions: (DO_NOT_USE_1, aceEditSession, pos, prefix, callback) => {
        const position = {
          lineNumber: pos.row + 1,
          column: pos.column + 1
        };
        const getAnnotationControls = () => {
          let customAnnotation;
          return {
            setAnnotation(text) {
              const annotations = aceEditSession.getAnnotations();
              customAnnotation = {
                text,
                row: pos.row,
                column: pos.column,
                type: 'warning'
              };
              aceEditSession.setAnnotations([...annotations, customAnnotation]);
            },
            removeAnnotation() {
              aceEditSession.setAnnotations(aceEditSession.getAnnotations().filter(a => a !== customAnnotation));
            }
          };
        };
        autocompleter(position, prefix, callback, getAnnotationControls());
      }
    }]);
  }
  destroy() {
    this.editor.destroy();
  }

  /**
   * Formats body of the request in the editor by removing the extra whitespaces at the beginning of lines,
   * And adds the correct indentation for each line
   * @param reqRange request range to indent
   */
  autoIndent(reqRange) {
    const session = this.editor.getSession();
    const mode = session.getMode();
    const startRow = reqRange.start.lineNumber;
    const endRow = reqRange.end.lineNumber;
    const tab = session.getTabString();
    for (let row = startRow; row <= endRow; row++) {
      let prevLineState = '';
      let prevLineIndent = '';
      if (row > 0) {
        prevLineState = session.getState(row - 1);
        const prevLine = session.getLine(row - 1);
        prevLineIndent = mode.getNextLineIndent(prevLineState, prevLine, tab);
      }
      const line = session.getLine(row);
      // @ts-ignore
      // Brace does not expose type definition for mode.$getIndent, though we have access to this method provided by the underlying Ace editor.
      // See https://github.com/ajaxorg/ace/blob/87ce087ed1cf20eeabe56fb0894e048d9bc9c481/lib/ace/mode/text.js#L259
      const currLineIndent = mode.$getIndent(line);
      if (prevLineIndent !== currLineIndent) {
        if (currLineIndent.length > 0) {
          // If current line has indentation, remove it.
          // Next we will add the correct indentation by looking at the previous line
          const range = new _AceRange(row, 0, row, currLineIndent.length);
          session.remove(range);
        }
        if (prevLineIndent.length > 0) {
          // If previous line has indentation, add indentation at the current line
          session.insert({
            row,
            column: 0
          }, prevLineIndent);
        }
      }

      // Lastly outdent any closing braces
      mode.autoOutdent(prevLineState, session, row);
    }
  }
  getAllFoldRanges() {
    const session = this.editor.getSession();
    // @ts-ignore
    // Brace does not expose type definition for session.getAllFolds, though we have access to this method provided by the underlying Ace editor.
    // See https://github.com/ajaxorg/ace/blob/13dc911dbc0ea31ca343d5744b3f472767458fc3/ace.d.ts#L82
    return session.getAllFolds().map(fold => fold.range);
  }
  addFoldsAtRanges(foldRanges) {
    const session = this.editor.getSession();
    foldRanges.forEach(range => {
      try {
        session.addFold('...', _AceRange.fromPoints(range.start, range.end));
      } catch (e) {
        // ignore the error if a fold fails
      }
    });
  }
}
exports.LegacyCoreEditor = LegacyCoreEditor;