"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Settings = Settings;
var _react = _interopRequireDefault(require("react"));
var _components = require("../components");
var _contexts = require("../contexts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const getAutocompleteDiff = (newSettings, prevSettings) => {
  return Object.keys(newSettings.autocomplete).filter(key => {
    // @ts-ignore
    return prevSettings.autocomplete[key] !== newSettings.autocomplete[key];
  });
};
function Settings({
  onClose,
  editorInstance
}) {
  const {
    services: {
      settings,
      autocompleteInfo
    }
  } = (0, _contexts.useServicesContext)();
  const dispatch = (0, _contexts.useEditorActionContext)();
  const refreshAutocompleteSettings = (settingsService, selectedSettings) => {
    autocompleteInfo.retrieve(settingsService, selectedSettings);
  };
  const fetchAutocompleteSettingsIfNeeded = (settingsService, newSettings, prevSettings) => {
    // We'll only retrieve settings if polling is on. The expectation here is that if the user
    // disables polling it's because they want manual control over the fetch request (possibly
    // because it's a very expensive request given their cluster and bandwidth). In that case,
    // they would be unhappy with any request that's sent automatically.
    if (newSettings.polling) {
      const autocompleteDiff = getAutocompleteDiff(newSettings, prevSettings);
      const isSettingsChanged = autocompleteDiff.length > 0;
      const isPollingChanged = prevSettings.polling !== newSettings.polling;
      if (isSettingsChanged) {
        // If the user has changed one of the autocomplete settings, then we'll fetch just the
        // ones which have changed.
        const changedSettings = autocompleteDiff.reduce((changedSettingsAccum, setting) => {
          changedSettingsAccum[setting] = newSettings.autocomplete[setting];
          return changedSettingsAccum;
        }, {});
        autocompleteInfo.retrieve(settingsService, {
          ...settingsService.getAutocomplete(),
          ...changedSettings
        });
      } else if (isPollingChanged && newSettings.polling) {
        // If the user has turned polling on, then we'll fetch all selected autocomplete settings.
        autocompleteInfo.retrieve(settingsService, settingsService.getAutocomplete());
      }
    }
  };
  const onSaveSettings = newSettings => {
    const prevSettings = settings.toJSON();
    fetchAutocompleteSettingsIfNeeded(settings, newSettings, prevSettings);

    // Update the new settings in localStorage
    settings.updateSettings(newSettings);

    // Let the rest of the application know settings have updated.
    dispatch({
      type: 'updateSettings',
      payload: newSettings
    });
    onClose();
  };
  return /*#__PURE__*/_react.default.createElement(_components.DevToolsSettingsModal, {
    onClose: onClose,
    onSaveSettings: onSaveSettings,
    refreshAutocompleteSettings: selectedSettings => refreshAutocompleteSettings(settings, selectedSettings),
    settings: settings.toJSON(),
    editorInstance: editorInstance
  });
}