"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.trackSentRequests = exports.stringifyRequest = exports.replaceRequestVariables = exports.getRequestStartLineNumber = exports.getRequestEndLineNumber = exports.getCurlRequest = exports.getAutoIndentedRequests = void 0;
var _es = require("../../../../../lib/es");
var _constants = require("./constants");
var _tokens_utils = require("./tokens_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/*
 * This function stringifies and normalizes the parsed request:
 * - the method is converted to upper case
 * - any trailing comments are removed from the url
 * - the request body is stringified from an object using JSON.stringify
 */
const stringifyRequest = parsedRequest => {
  var _parsedRequest$method, _parsedRequest$method2, _parsedRequest$data;
  const url = parsedRequest.url ? (0, _tokens_utils.removeTrailingWhitespaces)(parsedRequest.url) : '';
  const method = (_parsedRequest$method = (_parsedRequest$method2 = parsedRequest.method) === null || _parsedRequest$method2 === void 0 ? void 0 : _parsedRequest$method2.toUpperCase()) !== null && _parsedRequest$method !== void 0 ? _parsedRequest$method : '';
  const data = (_parsedRequest$data = parsedRequest.data) === null || _parsedRequest$data === void 0 ? void 0 : _parsedRequest$data.map(parsedData => JSON.stringify(parsedData, null, 2));
  return {
    url,
    method,
    data: data !== null && data !== void 0 ? data : []
  };
};

/*
 * This function replaces any variables with its values stored in localStorage.
 * For example 'GET ${exampleVariable1} -> 'GET _search'.
 */
exports.stringifyRequest = stringifyRequest;
const replaceRequestVariables = ({
  method,
  url,
  data
}, variables) => {
  return {
    method,
    url: replaceVariables(url, variables),
    data: data.map(dataObject => replaceVariables(dataObject, variables))
  };
};

/*
 * This function converts a request into a corresponding CURL command.
 */
exports.replaceRequestVariables = replaceRequestVariables;
const getCurlRequest = ({
  method,
  url,
  data
}, elasticsearchBaseUrl) => {
  const curlUrl = (0, _es.constructUrl)(elasticsearchBaseUrl, url);
  let curlRequest = `curl -X${method} "${curlUrl}" -H "kbn-xsrf: reporting"`;
  if (data.length > 0) {
    curlRequest += ` -H "Content-Type: application/json" -d'\n`;
    curlRequest += data.join('\n');
    curlRequest += "'";
  }
  return curlRequest;
};

/*
 * This function uses the telemetry to track requests sent via Console.
 */
exports.getCurlRequest = getCurlRequest;
const trackSentRequests = (requests, trackUiMetric) => {
  requests.map(({
    method,
    url
  }) => {
    const eventName = `${method}_${url}`;
    trackUiMetric.count(eventName);
  });
};

/*
 * This function converts the start offset value of the parsed request to a line number in the model
 */
exports.trackSentRequests = trackSentRequests;
const getRequestStartLineNumber = (parsedRequest, model) => {
  return model.getPositionAt(parsedRequest.startOffset).lineNumber;
};

/*
 * This function converts the end offset value of the parsed request to a line number in the model.
 * If there is no end offset (the parser was not able to parse this request completely),
 * then the last non-empty line is returned or the line before the next request.
 */
exports.getRequestStartLineNumber = getRequestStartLineNumber;
const getRequestEndLineNumber = (parsedRequest, model, index, parsedRequests) => {
  let endLineNumber;
  if (parsedRequest.endOffset) {
    // if the parser set an end offset for this request, then find the line number for it
    endLineNumber = model.getPositionAt(parsedRequest.endOffset).lineNumber;
  } else {
    // if no end offset, try to find the line before the next request starts
    const nextRequest = parsedRequests.at(index + 1);
    if (nextRequest) {
      const nextRequestStartLine = model.getPositionAt(nextRequest.startOffset).lineNumber;
      endLineNumber = nextRequestStartLine - 1;
    } else {
      // if there is no next request, take the last line of the model
      endLineNumber = model.getLineCount();
    }
  }
  // if the end line is empty, go up to find the first non-empty line
  let lineContent = model.getLineContent(endLineNumber).trim();
  while (!lineContent) {
    endLineNumber = endLineNumber - 1;
    lineContent = model.getLineContent(endLineNumber).trim();
  }
  return endLineNumber;
};

/*
 * Internal helpers
 */
exports.getRequestEndLineNumber = getRequestEndLineNumber;
const replaceVariables = (text, variables) => {
  if (_constants.variableTemplateRegex.test(text)) {
    text = text.replaceAll(_constants.variableTemplateRegex, (match, key) => {
      var _variable$value;
      const variable = variables.find(({
        name
      }) => name === key);
      return (_variable$value = variable === null || variable === void 0 ? void 0 : variable.value) !== null && _variable$value !== void 0 ? _variable$value : match;
    });
  }
  return text;
};
const containsComments = text => {
  return text.indexOf('//') >= 0 || text.indexOf('/*') >= 0;
};

/**
 * This function takes a string containing unformatted Console requests and
 * returns a text in which the requests are auto-indented.
 * @param requests The list of {@link AdjustedParsedRequest} that are in the selected text in the editor.
 * @param selectedText The selected text in the editor.
 * @param allText The whole text input in the editor.
 */
const getAutoIndentedRequests = (requests, selectedText, allText) => {
  const selectedTextLines = selectedText.split(`\n`);
  const allTextLines = allText.split(`\n`);
  const formattedTextLines = [];
  let currentLineIndex = 0;
  let currentRequestIndex = 0;
  while (currentLineIndex < selectedTextLines.length) {
    const request = requests[currentRequestIndex];
    // Check if the current line is the start of the next request
    if (request && selectedTextLines[currentLineIndex] === allTextLines[request.startLineNumber - 1]) {
      // Start of a request
      const requestLines = allTextLines.slice(request.startLineNumber - 1, request.endLineNumber);
      if (requestLines.some(line => containsComments(line))) {
        // If request has comments, add it as it is - without formatting
        // TODO: Format requests with comments
        formattedTextLines.push(...requestLines);
      } else {
        // If no comments, add stringified parsed request
        const stringifiedRequest = stringifyRequest(request);
        const firstLine = stringifiedRequest.method + ' ' + stringifiedRequest.url;
        formattedTextLines.push(firstLine);
        if (stringifiedRequest.data && stringifiedRequest.data.length > 0) {
          formattedTextLines.push(...stringifiedRequest.data);
        }
      }
      currentLineIndex = currentLineIndex + requestLines.length;
      currentRequestIndex++;
    } else {
      // Current line is a comment or whitespaces
      // Trim white spaces and add it to the formatted text
      formattedTextLines.push(selectedTextLines[currentLineIndex].trim());
      currentLineIndex++;
    }
  }
  return formattedTextLines.join('\n');
};
exports.getAutoIndentedRequests = getAutoIndentedRequests;