"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.shouldTriggerSuggestions = exports.isEmptyOrDoubleQuote = exports.getUrlPathCompletionItems = exports.getUrlParamsCompletionItems = exports.getMethodCompletionItems = exports.getDocumentationLinkFromAutocomplete = exports.getBodyCompletionItems = void 0;
var _monaco = require("@kbn/monaco");
var _kb = require("../../../../../lib/kb");
var _engine = require("../../../../../lib/autocomplete/engine");
var _tokens_utils = require("./tokens_utils");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/*
 * This function initializes the autocomplete context for the request
 * and returns a documentation link from the endpoint object
 * with the branch in the url replaced by the current version "docLinkVersion"
 */
const getDocumentationLinkFromAutocomplete = (request, docLinkVersion) => {
  // get the url parts from the request url
  const {
    urlPathTokens
  } = (0, _tokens_utils.parseUrl)(request.url);
  // remove the last token, if it's empty
  if (!urlPathTokens[urlPathTokens.length - 1]) {
    urlPathTokens.pop();
  }
  // add the end of url token
  urlPathTokens.push(_constants.END_OF_URL_TOKEN);
  const {
    endpoint
  } = populateContextForMethodAndUrl(request.method, urlPathTokens);
  if (endpoint && endpoint.documentation && endpoint.documentation.indexOf('http') !== -1) {
    return endpoint.documentation.replace('/master/', `/${docLinkVersion}/`).replace('/current/', `/${docLinkVersion}/`).replace('/{branch}/', `/${docLinkVersion}/`);
  }
  return null;
};
/*
 * Helper function that filters out suggestions without a name.
 */
exports.getDocumentationLinkFromAutocomplete = getDocumentationLinkFromAutocomplete;
const filterTermsWithoutName = terms => terms.filter(term => term.name !== undefined);

/*
 * This function returns an array of completion items for the request method
 */
const autocompleteMethods = ['GET', 'PUT', 'POST', 'DELETE', 'HEAD', 'PATCH'];
const getMethodCompletionItems = (model, position) => {
  // get the word before suggestions to replace when selecting a suggestion from the list
  const wordUntilPosition = model.getWordUntilPosition(position);
  return autocompleteMethods.map(method => ({
    label: method,
    insertText: method,
    detail: _constants.i18nTexts.method,
    // only used to configure the icon
    kind: _monaco.monaco.languages.CompletionItemKind.Constant,
    range: {
      // replace the whole word with the suggestion
      startColumn: wordUntilPosition.startColumn,
      startLineNumber: position.lineNumber,
      endColumn: position.column,
      endLineNumber: position.lineNumber
    }
  }));
};

/*
 * This function initializes the autocomplete context for the provided method and url token path.
 */
exports.getMethodCompletionItems = getMethodCompletionItems;
const populateContextForMethodAndUrl = (method, urlTokenPath) => {
  // get autocomplete components for the request method
  const components = (0, _kb.getTopLevelUrlCompleteComponents)(method);
  // this object will contain the information later, it needs to be initialized with some data
  // similar to the old ace editor context
  const context = {
    method,
    urlTokenPath
  };

  // mutate the context object and put the autocomplete information there
  (0, _engine.populateContext)(urlTokenPath, context, undefined, true, components);
  return context;
};

/*
 * This function returns an array of completion items for the request method and the url path
 */
const getUrlPathCompletionItems = (model, position) => {
  const {
    lineNumber,
    column
  } = position;
  // get the content of the line up until the current position
  const lineContent = model.getValueInRange({
    startLineNumber: lineNumber,
    startColumn: 1,
    endLineNumber: lineNumber,
    endColumn: column
  });

  // get the method and previous url parts for context
  const {
    method,
    urlPathTokens
  } = (0, _tokens_utils.parseLine)(lineContent);
  // remove the last token that is either empty if the url has like "_search/" as the last char
  // or it's a word that need to be replaced with autocomplete suggestions like "_search/s"
  urlPathTokens.pop();
  const {
    autoCompleteSet
  } = populateContextForMethodAndUrl(method, urlPathTokens);
  const wordUntilPosition = model.getWordUntilPosition(position);
  const range = {
    startLineNumber: position.lineNumber,
    // replace the whole word with the suggestion
    startColumn: lineContent.endsWith('.') ?
    // if there is a dot at the end of the content, it's ignored in the wordUntilPosition
    wordUntilPosition.startColumn - 1 : wordUntilPosition.startColumn,
    endLineNumber: position.lineNumber,
    endColumn: position.column
  };
  if (autoCompleteSet && autoCompleteSet.length > 0) {
    return filterTermsWithoutName(autoCompleteSet)
    // map autocomplete items to completion items
    .map(item => {
      var _item$meta;
      return {
        label: item.name + '',
        insertText: item.name + '',
        detail: (_item$meta = item.meta) !== null && _item$meta !== void 0 ? _item$meta : _constants.i18nTexts.endpoint,
        // the kind is only used to configure the icon
        kind: _monaco.monaco.languages.CompletionItemKind.Constant,
        range
      };
    });
  }
  return [];
};

/*
 * This function returns an array of completion items for the url params
 */
exports.getUrlPathCompletionItems = getUrlPathCompletionItems;
const getUrlParamsCompletionItems = (model, position) => {
  var _context$endpoint;
  const {
    lineNumber,
    column
  } = position;
  // get the content of the line up until the current position
  const lineContent = model.getValueInRange({
    startLineNumber: lineNumber,
    startColumn: 1,
    endLineNumber: lineNumber,
    endColumn: column
  });

  // get the method and previous url parts for context
  const {
    method,
    urlPathTokens,
    urlParamsTokens
  } = (0, _tokens_utils.parseLine)(lineContent);
  urlPathTokens.push(_constants.END_OF_URL_TOKEN);
  const context = populateContextForMethodAndUrl(method, urlPathTokens);
  const urlParamsComponents = (_context$endpoint = context.endpoint) === null || _context$endpoint === void 0 ? void 0 : _context$endpoint.paramsAutocomplete.getTopLevelComponents(method);
  const currentUrlParamToken = urlParamsTokens.pop();
  // check if we are at the param name or the param value
  const urlParamTokenPath = [];
  // if there are 2 tokens in the current url param, then we have the name and the value of the param
  if (currentUrlParamToken && currentUrlParamToken.length > 1) {
    urlParamTokenPath.push(currentUrlParamToken[0]);
  }
  (0, _engine.populateContext)(urlParamTokenPath, context, undefined, true, urlParamsComponents);
  if (context.autoCompleteSet && context.autoCompleteSet.length > 0) {
    const wordUntilPosition = model.getWordUntilPosition(position);
    const range = {
      startLineNumber: position.lineNumber,
      // replace the whole word with the suggestion
      startColumn: wordUntilPosition.startColumn,
      endLineNumber: position.lineNumber,
      endColumn: position.column
    };
    return filterTermsWithoutName(context.autoCompleteSet)
    // map autocomplete items to completion items
    .map(item => {
      var _item$meta2;
      return {
        label: item.name + '',
        insertText: item.name + '',
        detail: (_item$meta2 = item.meta) !== null && _item$meta2 !== void 0 ? _item$meta2 : _constants.i18nTexts.param,
        // the kind is only used to configure the icon
        kind: _monaco.monaco.languages.CompletionItemKind.Constant,
        range
      };
    });
  }
  return [];
};

/*
 * This function returns an array of completion items for the request body params
 */
exports.getUrlParamsCompletionItems = getUrlParamsCompletionItems;
const getBodyCompletionItems = async (model, position, requestStartLineNumber, editor) => {
  var _context$asyncResults, _context$autoComplete;
  const {
    lineNumber,
    column
  } = position;

  // get the content on the method+url line
  const lineContent = model.getLineContent(requestStartLineNumber);
  // get the method and previous url parts for context
  const {
    method,
    urlPathTokens
  } = (0, _tokens_utils.parseLine)(lineContent);
  urlPathTokens.push(_constants.END_OF_URL_TOKEN);
  const context = populateContextForMethodAndUrl(method, urlPathTokens);

  // get the content of the request body up until this position
  const bodyRange = {
    startLineNumber: requestStartLineNumber + 1,
    startColumn: 1,
    endLineNumber: lineNumber,
    endColumn: column
  };
  const bodyContentBeforePosition = model.getValueInRange(bodyRange);
  const bodyTokens = (0, _tokens_utils.parseBody)(bodyContentBeforePosition);
  // needed for scope linking + global term resolving
  context.endpointComponentResolver = _kb.getEndpointBodyCompleteComponents;
  context.globalComponentResolver = _kb.getGlobalAutocompleteComponents;
  let components;
  if (context.endpoint) {
    components = context.endpoint.bodyAutocompleteRootComponents;
  } else {
    components = (0, _kb.getUnmatchedEndpointComponents)();
  }
  context.editor = editor;
  context.requestStartRow = requestStartLineNumber;
  (0, _engine.populateContext)(bodyTokens, context, editor, true, components);
  if (!context) {
    return [];
  }
  // loading async suggestions
  if ((_context$asyncResults = context.asyncResultsState) !== null && _context$asyncResults !== void 0 && _context$asyncResults.isLoading && context.asyncResultsState) {
    const results = await context.asyncResultsState.results;
    return getSuggestions(model, position, results, context, bodyContentBeforePosition);
  }
  return getSuggestions(model, position, (_context$autoComplete = context.autoCompleteSet) !== null && _context$autoComplete !== void 0 ? _context$autoComplete : [], context, bodyContentBeforePosition);
};
exports.getBodyCompletionItems = getBodyCompletionItems;
const getSuggestions = (model, position, autocompleteSet, context, bodyContentBeforePosition) => {
  const wordUntilPosition = model.getWordUntilPosition(position);
  const lineContentAfterPosition = model.getValueInRange({
    startLineNumber: position.lineNumber,
    startColumn: position.column,
    endLineNumber: position.lineNumber,
    endColumn: model.getLineMaxColumn(position.lineNumber)
  });
  // if the rest of the line is empty or there is only "
  // then template can be inserted, otherwise only name
  context.addTemplate = isEmptyOrDoubleQuote(lineContentAfterPosition);

  // if there is " after the cursor, include it in the insert range
  let endColumn = position.column;
  if (lineContentAfterPosition.startsWith('"')) {
    endColumn = endColumn + 1;
  }
  const range = {
    startLineNumber: position.lineNumber,
    // replace the whole word with the suggestion
    startColumn: wordUntilPosition.startColumn,
    endLineNumber: position.lineNumber,
    endColumn
  };
  return filterTermsWithoutName(autocompleteSet)
  // map autocomplete items to completion items
  .map(item => {
    const suggestion = {
      // convert name to a string
      label: item.name + '',
      insertText: getInsertText(item, bodyContentBeforePosition, context),
      detail: _constants.i18nTexts.api,
      // the kind is only used to configure the icon
      kind: _monaco.monaco.languages.CompletionItemKind.Constant,
      range,
      insertTextRules: _monaco.monaco.languages.CompletionItemInsertTextRule.InsertAsSnippet
    };
    return suggestion;
  });
};
const getInsertText = ({
  name,
  insertValue,
  template,
  value
}, bodyContent, context) => {
  if (name === undefined) {
    return '';
  }
  let insertText = '';
  if (typeof name === 'string') {
    insertText = bodyContent.endsWith('"') ? '' : '"';
    if (insertValue && insertValue !== '{' && insertValue !== '[') {
      insertText += `${insertValue}"`;
    } else {
      insertText += `${name}"`;
    }
  } else {
    insertText = name + '';
  }

  // check if there is template to add
  const conditionalTemplate = getConditionalTemplate(name, bodyContent, context.endpoint);
  if (conditionalTemplate) {
    template = conditionalTemplate;
  }
  if (template !== undefined && context.addTemplate) {
    let templateLines;
    const {
      __raw,
      value: templateValue
    } = template;
    if (__raw && templateValue) {
      templateLines = templateValue.split(_constants.newLineRegex);
    } else {
      templateLines = JSON.stringify(template, null, 2).split(_constants.newLineRegex);
    }
    insertText += ': ' + templateLines.join('\n');
  } else if (value === '{') {
    insertText += '{}';
  } else if (value === '[') {
    insertText += '[]';
  }
  // the string $0 is used to move the cursor between empty curly/square brackets
  if (insertText.endsWith('{}')) {
    insertText = insertText.substring(0, insertText.length - 2) + '{$0}';
  }
  if (insertText.endsWith('[]')) {
    insertText = insertText.substring(0, insertText.length - 2) + '[$0]';
  }
  return insertText;
};
const getConditionalTemplate = (name, bodyContent, endpoint) => {
  if (typeof name !== 'string' || !endpoint || !endpoint.data_autocomplete_rules) {
    return;
  }
  // get the autocomplete rules for the request body
  const {
    data_autocomplete_rules: autocompleteRules
  } = endpoint;
  // get the rules for this property name
  const rules = autocompleteRules[name];
  // check if the rules have "__one_of" property
  if (!rules || typeof rules !== 'object' || !('__one_of' in rules)) {
    return;
  }
  const oneOfRules = rules.__one_of;
  // try to match one of the rules to the body content
  const matchedRule = oneOfRules.find(rule => {
    if (rule.__condition && rule.__condition.lines_regex) {
      return new RegExp(rule.__condition.lines_regex, 'm').test(bodyContent);
    }
    return false;
  });
  // use the template from the matched rule
  if (matchedRule && matchedRule.__template) {
    return matchedRule.__template;
  }
};

/*
 * This function checks the content of the line before the cursor and decides if the autocomplete
 * suggestions should be triggered
 */
const shouldTriggerSuggestions = lineContent => {
  return _constants.methodWhitespaceRegex.test(lineContent) || _constants.methodWithUrlRegex.test(lineContent) || _constants.propertyNameRegex.test(lineContent) || _constants.propertyValueRegex.test(lineContent);
};

/*
 * This function checks if the content of the line after the cursor is either empty
 * or it only has a double quote.
 */
exports.shouldTriggerSuggestions = shouldTriggerSuggestions;
const isEmptyOrDoubleQuote = lineContent => {
  lineContent = lineContent.trim();
  return !lineContent || lineContent === '"';
};
exports.isEmptyOrDoubleQuote = isEmptyOrDoubleQuote;