"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MonacoEditorActionsProvider = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _monaco = require("@kbn/monaco");
var _i18n = require("@kbn/i18n");
var _reactKibanaMount = require("@kbn/react-kibana-mount");
var _history = require("../../../../services/history");
var _constants = require("../../../../../common/constants");
var _services = require("../../../../services");
var _hooks = require("../../../hooks");
var _utils = require("./utils");
var _storage_quota_error = require("../../../components/storage_quota_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const AUTO_INDENTATION_ACTION_LABEL = 'Apply indentations';
const TRIGGER_SUGGESTIONS_ACTION_LABEL = 'Trigger suggestions';
const TRIGGER_SUGGESTIONS_HANDLER_ID = 'editor.action.triggerSuggest';
const DEBOUNCE_HIGHLIGHT_WAIT_MS = 200;
const DEBOUNCE_AUTOCOMPLETE_WAIT_MS = 500;
class MonacoEditorActionsProvider {
  constructor(editor, setEditorActionsCss) {
    (0, _defineProperty2.default)(this, "parsedRequestsProvider", void 0);
    (0, _defineProperty2.default)(this, "highlightedLines", void 0);
    this.editor = editor;
    this.setEditorActionsCss = setEditorActionsCss;
    this.parsedRequestsProvider = (0, _monaco.getParsedRequestsProvider)(this.editor.getModel());
    this.highlightedLines = this.editor.createDecorationsCollection();
    this.editor.focus();
    const debouncedHighlightRequests = (0, _lodash.debounce)(() => this.highlightRequests(), DEBOUNCE_HIGHLIGHT_WAIT_MS, {
      leading: true
    });
    debouncedHighlightRequests();
    const debouncedTriggerSuggestions = (0, _lodash.debounce)(() => {
      this.triggerSuggestions();
    }, DEBOUNCE_AUTOCOMPLETE_WAIT_MS, {
      leading: false,
      trailing: true
    });

    // init all listeners
    editor.onDidChangeCursorPosition(async event => {
      await debouncedHighlightRequests();
    });
    editor.onDidScrollChange(async event => {
      await debouncedHighlightRequests();
    });
    editor.onDidChangeCursorSelection(async event => {
      await debouncedHighlightRequests();
    });
    editor.onDidContentSizeChange(async event => {
      await debouncedHighlightRequests();
    });
    editor.onKeyUp(event => {
      // trigger autocomplete on backspace
      if (event.keyCode === _monaco.monaco.KeyCode.Backspace) {
        debouncedTriggerSuggestions();
      }
    });
  }
  updateEditorActions(lineNumber) {
    // if no request is currently selected, hide the actions buttons
    if (!lineNumber) {
      this.setEditorActionsCss({
        visibility: 'hidden'
      });
    } else {
      // if a request is selected, the actions buttons are placed at lineNumberOffset - scrollOffset
      const offset = this.editor.getTopForLineNumber(lineNumber) - this.editor.getScrollTop();
      this.setEditorActionsCss({
        visibility: 'visible',
        top: offset
      });
    }
  }
  async highlightRequests() {
    // get the requests in the selected range
    const parsedRequests = await this.getSelectedParsedRequests();
    // if any requests are selected, highlight the lines and update the position of actions buttons
    if (parsedRequests.length > 0) {
      var _this$editor$getModel, _this$editor$getModel2;
      // display the actions buttons on the 1st line of the 1st selected request
      const selectionStartLineNumber = parsedRequests[0].startLineNumber;
      this.updateEditorActions(selectionStartLineNumber);
      // highlight the lines from the 1st line of the first selected request
      // to the last line of the last selected request
      const selectionEndLineNumber = parsedRequests[parsedRequests.length - 1].endLineNumber;
      const selectedRange = new _monaco.monaco.Range(selectionStartLineNumber, 1, selectionEndLineNumber, (_this$editor$getModel = (_this$editor$getModel2 = this.editor.getModel()) === null || _this$editor$getModel2 === void 0 ? void 0 : _this$editor$getModel2.getLineMaxColumn(selectionEndLineNumber)) !== null && _this$editor$getModel !== void 0 ? _this$editor$getModel : 1);
      this.highlightedLines.set([{
        range: selectedRange,
        options: {
          isWholeLine: true,
          className: _utils.SELECTED_REQUESTS_CLASSNAME
        }
      }]);
    } else {
      // if no requests are selected, hide actions buttons and remove highlighted lines
      this.updateEditorActions();
      this.highlightedLines.clear();
    }
  }
  async getSelectedParsedRequests() {
    const model = this.editor.getModel();
    const selection = this.editor.getSelection();
    if (!model || !selection) {
      return Promise.resolve([]);
    }
    const {
      startLineNumber,
      endLineNumber
    } = selection;
    return this.getRequestsBetweenLines(model, startLineNumber, endLineNumber);
  }
  async getRequestsBetweenLines(model, startLineNumber, endLineNumber) {
    const parsedRequests = await this.parsedRequestsProvider.getRequests();
    const selectedRequests = [];
    for (const [index, parsedRequest] of parsedRequests.entries()) {
      const requestStartLineNumber = (0, _utils.getRequestStartLineNumber)(parsedRequest, model);
      const requestEndLineNumber = (0, _utils.getRequestEndLineNumber)(parsedRequest, model, index, parsedRequests);
      if (requestStartLineNumber > endLineNumber) {
        // request is past the selection, no need to check further requests
        break;
      }
      if (requestEndLineNumber < startLineNumber) {
        // request is before the selection, do nothing
      } else {
        // request is selected
        selectedRequests.push({
          ...parsedRequest,
          startLineNumber: requestStartLineNumber,
          endLineNumber: requestEndLineNumber
        });
      }
    }
    return selectedRequests;
  }
  async getRequests() {
    const parsedRequests = await this.getSelectedParsedRequests();
    const stringifiedRequests = parsedRequests.map(parsedRequest => (0, _utils.stringifyRequest)(parsedRequest));
    // get variables values
    const variables = (0, _services.getStorage)().get(_services.StorageKeys.VARIABLES, _constants.DEFAULT_VARIABLES);
    return stringifiedRequests.map(request => (0, _utils.replaceRequestVariables)(request, variables));
  }
  async getCurl(elasticsearchBaseUrl) {
    const requests = await this.getRequests();
    const curlRequests = requests.map(request => (0, _utils.getCurlRequest)(request, elasticsearchBaseUrl));
    return curlRequests.join('\n');
  }
  async sendRequests(dispatch, context) {
    const {
      services: {
        notifications,
        trackUiMetric,
        http,
        settings,
        history,
        autocompleteInfo
      },
      ...startServices
    } = context;
    const {
      toasts
    } = notifications;
    try {
      const requests = await this.getRequests();
      if (!requests.length) {
        toasts.add(_i18n.i18n.translate('console.notification.error.noRequestSelectedTitle', {
          defaultMessage: 'No request selected. Select a request by placing the cursor inside it.'
        }));
        return;
      }
      dispatch({
        type: 'sendRequest',
        payload: undefined
      });

      // track the requests
      setTimeout(() => (0, _utils.trackSentRequests)(requests, trackUiMetric), 0);
      const results = await (0, _hooks.sendRequest)({
        http,
        requests
      });
      let saveToHistoryError;
      const isHistoryEnabled = settings.getIsHistoryEnabled();
      if (isHistoryEnabled) {
        results.forEach(({
          request: {
            path,
            method,
            data
          }
        }) => {
          try {
            history.addToHistory(path, method, data);
          } catch (e) {
            // Grab only the first error
            if (!saveToHistoryError) {
              saveToHistoryError = e;
            }
          }
        });
        if (saveToHistoryError) {
          const errorTitle = _i18n.i18n.translate('console.notification.error.couldNotSaveRequestTitle', {
            defaultMessage: 'Could not save request to Console history.'
          });
          if ((0, _history.isQuotaExceededError)(saveToHistoryError)) {
            const toast = notifications.toasts.addWarning({
              title: _i18n.i18n.translate('console.notification.error.historyQuotaReachedMessage', {
                defaultMessage: 'Request history is full. Clear the console history or disable saving new requests.'
              }),
              text: (0, _reactKibanaMount.toMountPoint)((0, _storage_quota_error.StorageQuotaError)({
                onClearHistory: () => {
                  history.clearHistory();
                  notifications.toasts.remove(toast);
                },
                onDisableSavingToHistory: () => {
                  settings.setIsHistoryEnabled(false);
                  notifications.toasts.remove(toast);
                }
              }), startServices)
            });
          } else {
            // Best effort, but still notify the user.
            notifications.toasts.addError(saveToHistoryError, {
              title: errorTitle
            });
          }
        }
      }
      const polling = settings.getPolling();
      if (polling) {
        // If the user has submitted a request against ES, something in the fields, indices, aliases,
        // or templates may have changed, so we'll need to update this data. Assume that if
        // the user disables polling they're trying to optimize performance or otherwise
        // preserve resources, so they won't want this request sent either.
        autocompleteInfo.retrieve(settings, settings.getAutocomplete());
      }
      dispatch({
        type: 'requestSuccess',
        payload: {
          data: results
        }
      });
    } catch (e) {
      if (e !== null && e !== void 0 && e.response) {
        dispatch({
          type: 'requestFail',
          payload: e
        });
      } else {
        dispatch({
          type: 'requestFail',
          payload: undefined
        });
        toasts.addError(e, {
          title: _i18n.i18n.translate('console.notification.error.unknownErrorTitle', {
            defaultMessage: 'Unknown Request Error'
          })
        });
      }
    }
  }
  async getDocumentationLink(docLinkVersion) {
    const requests = await this.getRequests();
    if (requests.length < 1) {
      return null;
    }
    const request = requests[0];
    return (0, _utils.getDocumentationLinkFromAutocomplete)(request, docLinkVersion);
  }
  async getAutocompleteType(model, {
    lineNumber,
    column
  }) {
    // get the current request on this line
    const currentRequests = await this.getRequestsBetweenLines(model, lineNumber, lineNumber);
    const currentRequest = currentRequests.at(0);
    // if there is no request, suggest method
    if (!currentRequest) {
      return _utils.AutocompleteType.METHOD;
    }

    // if on the 1st line of the request, suggest method, url or url_params depending on the content
    const {
      startLineNumber: requestStartLineNumber
    } = currentRequest;
    if (lineNumber === requestStartLineNumber) {
      // get the content on the line up until the position
      const lineContent = model.getValueInRange({
        startLineNumber: lineNumber,
        startColumn: 1,
        endLineNumber: lineNumber,
        endColumn: column
      });
      const lineTokens = (0, _utils.getLineTokens)(lineContent);
      // if there is 1 or fewer tokens, suggest method
      if (lineTokens.length <= 1) {
        return _utils.AutocompleteType.METHOD;
      }
      // if there are 2 tokens, look at the 2nd one and suggest path or url_params
      if (lineTokens.length === 2) {
        const token = lineTokens[1];
        if ((0, _utils.containsUrlParams)(token)) {
          return _utils.AutocompleteType.URL_PARAMS;
        }
        return _utils.AutocompleteType.PATH;
      }
      // if more than 2 tokens, no suggestions
      return null;
    }

    // if not on the 1st line of the request, suggest request body

    return _utils.AutocompleteType.BODY;
  }
  async getSuggestions(model, position, context) {
    // determine autocomplete type
    const autocompleteType = await this.getAutocompleteType(model, position);
    if (!autocompleteType) {
      return {
        suggestions: []
      };
    }
    if (autocompleteType === _utils.AutocompleteType.METHOD) {
      return {
        // suggest all methods, the editor will filter according to the input automatically
        suggestions: (0, _utils.getMethodCompletionItems)(model, position)
      };
    }
    if (autocompleteType === _utils.AutocompleteType.PATH) {
      return {
        suggestions: (0, _utils.getUrlPathCompletionItems)(model, position)
      };
    }
    if (autocompleteType === _utils.AutocompleteType.URL_PARAMS) {
      return {
        suggestions: (0, _utils.getUrlParamsCompletionItems)(model, position)
      };
    }
    if (autocompleteType === _utils.AutocompleteType.BODY) {
      // suggestions only when triggered by " or keyboard
      if (context.triggerCharacter && context.triggerCharacter !== '"') {
        return {
          suggestions: []
        };
      }
      const requests = await this.getRequestsBetweenLines(model, position.lineNumber, position.lineNumber);
      const requestStartLineNumber = requests[0].startLineNumber;
      const suggestions = await (0, _utils.getBodyCompletionItems)(model, position, requestStartLineNumber, this);
      return {
        suggestions
      };
    }
    return {
      suggestions: []
    };
  }
  async provideCompletionItems(model, position, context, token) {
    return this.getSuggestions(model, position, context);
  }

  /*
   * This function inserts a request from the history into the editor
   */
  async restoreRequestFromHistory(request) {
    const model = this.editor.getModel();
    if (!model) {
      return;
    }
    let position = this.editor.getPosition();
    const requests = await this.getSelectedParsedRequests();
    let prefix = '';
    let suffix = '';
    // if there are requests at the cursor/selection, insert either before or after
    if (requests.length > 0) {
      // if on the 1st line of the 1st request, insert at the beginning of that line
      if (position && position.lineNumber === requests[0].startLineNumber) {
        position = {
          column: 1,
          lineNumber: position.lineNumber
        };
        suffix = '\n';
      } else {
        // otherwise insert at the end of the last line of the last request
        const lastLineNumber = requests[requests.length - 1].endLineNumber;
        position = {
          column: model.getLineMaxColumn(lastLineNumber),
          lineNumber: lastLineNumber
        };
        prefix = '\n';
      }
    } else {
      // if not inside a request, insert the request at the cursor line
      if (position) {
        // insert at the beginning of the cursor line
        position = {
          lineNumber: position.lineNumber,
          column: 1
        };
      } else {
        // otherwise insert on line 1
        position = {
          lineNumber: 1,
          column: 1
        };
      }
      suffix = '\n';
    }
    const edit = {
      range: {
        startLineNumber: position.lineNumber,
        startColumn: position.column,
        endLineNumber: position.lineNumber,
        endColumn: position.column
      },
      text: prefix + request + suffix,
      forceMoveMarkers: true
    };
    this.editor.executeEdits('restoreFromHistory', [edit]);
  }

  /*
  This function returns the text in the provided range.
  If no range is provided, it returns all text in the editor.
  */
  getTextInRange(selectionRange) {
    const model = this.editor.getModel();
    if (!model) {
      return '';
    }
    if (selectionRange) {
      const {
        startLineNumber,
        startColumn,
        endLineNumber,
        endColumn
      } = selectionRange;
      return model.getValueInRange({
        startLineNumber,
        startColumn,
        endLineNumber,
        endColumn
      });
    }
    // If no range is provided, return all text in the editor
    return model.getValue();
  }

  /**
   * This function applies indentations to the request in the selected text.
   */
  async autoIndent() {
    var _this$editor$getModel3, _this$editor$getModel4;
    const parsedRequests = await this.getSelectedParsedRequests();
    const selectionStartLineNumber = parsedRequests[0].startLineNumber;
    const selectionEndLineNumber = parsedRequests[parsedRequests.length - 1].endLineNumber;
    const selectedRange = new _monaco.monaco.Range(selectionStartLineNumber, 1, selectionEndLineNumber, (_this$editor$getModel3 = (_this$editor$getModel4 = this.editor.getModel()) === null || _this$editor$getModel4 === void 0 ? void 0 : _this$editor$getModel4.getLineMaxColumn(selectionEndLineNumber)) !== null && _this$editor$getModel3 !== void 0 ? _this$editor$getModel3 : 1);
    if (parsedRequests.length < 1) {
      return;
    }
    const selectedText = this.getTextInRange(selectedRange);
    const allText = this.getTextInRange();
    const autoIndentedText = (0, _utils.getAutoIndentedRequests)(parsedRequests, selectedText, allText);
    this.editor.executeEdits(AUTO_INDENTATION_ACTION_LABEL, [{
      range: selectedRange,
      text: autoIndentedText
    }]);
  }

  /**
   * This function moves the cursor to the previous request edge (start/end line).
   * If the cursor is inside a request, it is moved to the start line of this request.
   * If there are no requests before the cursor, it is moved at the first line in the editor.
   */
  async moveToPreviousRequestEdge() {
    const currentPosition = this.editor.getPosition();
    const model = this.editor.getModel();
    if (!currentPosition || !model) {
      return;
    }
    const {
      lineNumber: currentLineNumber
    } = currentPosition;
    // Get all requests before the current line
    const requestsBefore = await this.getRequestsBetweenLines(model, 1, currentLineNumber - 1);
    if (requestsBefore.length === 0) {
      // If no requests before current line, set position to first line
      this.editor.setPosition({
        lineNumber: 1,
        column: 1
      });
      return;
    }
    const lastRequestBefore = requestsBefore[requestsBefore.length - 1];
    if (lastRequestBefore.endLineNumber < currentLineNumber) {
      this.editor.setPosition({
        lineNumber: lastRequestBefore.endLineNumber,
        column: 1
      });
    } else {
      // If the end line of the request is after the current line, then the cursor is inside the request
      // The previous request edge is the start line of the request
      this.editor.setPosition({
        lineNumber: lastRequestBefore.startLineNumber,
        column: 1
      });
    }
  }

  /**
   * This function moves the cursor to the next request edge.
   * If the cursor is inside a request, it is moved to the end line of this request.
   * If there are no requests after the cursor, it is moved at the last line in the editor.
   */
  async moveToNextRequestEdge() {
    const currentPosition = this.editor.getPosition();
    const model = this.editor.getModel();
    if (!currentPosition || !model) {
      return;
    }
    const {
      lineNumber: currentLineNumber
    } = currentPosition;
    // Get all requests before the current line
    const requestsAfter = await this.getRequestsBetweenLines(model, currentLineNumber + 1, model.getLineCount());
    if (requestsAfter.length === 0) {
      // If no requests after current line, set position to last line
      this.editor.setPosition({
        lineNumber: model.getLineCount(),
        column: 1
      });
      return;
    }
    const firstRequestAfter = requestsAfter[0];
    if (firstRequestAfter.startLineNumber > currentLineNumber) {
      this.editor.setPosition({
        lineNumber: firstRequestAfter.startLineNumber,
        column: 1
      });
    } else {
      // If the start line of the request is before the current line, then the cursor is inside the request
      // The next request edge is the end line of the request
      this.editor.setPosition({
        lineNumber: firstRequestAfter.endLineNumber,
        column: 1
      });
    }
  }

  /*
   * This function is to get an array of line contents
   * from startLine to endLine including both line numbers
   */
  getLines(startLine, endLine) {
    const model = this.editor.getModel();
    if (!model) {
      return [];
    }
    // range returns an array not including the end of the range, so we need to add 1
    return (0, _lodash.range)(startLine, endLine + 1).map(lineNumber => model.getLineContent(lineNumber));
  }

  /*
   * This function returns the current position of the cursor
   */
  getCurrentPosition() {
    var _this$editor$getPosit;
    return (_this$editor$getPosit = this.editor.getPosition()) !== null && _this$editor$getPosit !== void 0 ? _this$editor$getPosit : {
      lineNumber: 1,
      column: 1
    };
  }
  triggerSuggestions() {
    const model = this.editor.getModel();
    const position = this.editor.getPosition();
    if (!model || !position) {
      return;
    }
    const lineContentBefore = model.getValueInRange({
      startLineNumber: position.lineNumber,
      startColumn: 1,
      endLineNumber: position.lineNumber,
      endColumn: position.column
    });
    // if the line is empty or it matches specified regex, trigger suggestions
    if (!lineContentBefore.trim() || (0, _utils.shouldTriggerSuggestions)(lineContentBefore)) {
      this.editor.trigger(TRIGGER_SUGGESTIONS_ACTION_LABEL, TRIGGER_SUGGESTIONS_HANDLER_ID, {});
    }
  }
}
exports.MonacoEditorActionsProvider = MonacoEditorActionsProvider;