"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSetInitialValue = exports.readLoadFromParam = void 0;
var _lodash = require("lodash");
var _queryString = require("query-string");
var _lzString = require("lz-string");
var _i18n = require("@kbn/i18n");
var _react = require("react");
var _constants = require("../../../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Util function for reading the load_from parameter from the current url.
 */

const readLoadFromParam = () => {
  const [, queryString] = (window.location.hash || window.location.search || '').split('?');
  const queryParams = (0, _queryString.parse)(queryString || '', {
    sort: false
  });
  return queryParams.load_from;
};

/**
 * Hook that sets the initial value in the Console editor.
 *
 * @param params The {@link SetInitialValueParams} to use.
 */
exports.readLoadFromParam = readLoadFromParam;
const useSetInitialValue = params => {
  const {
    initialTextValue,
    setValue,
    toasts
  } = params;
  (0, _react.useEffect)(() => {
    const loadBufferFromRemote = async url => {
      if (/^https?:\/\//.test(url)) {
        // Check if this is a valid URL
        try {
          new URL(url);
        } catch (e) {
          return;
        }
        // Parse the URL to avoid issues with spaces and other special characters.
        const parsedURL = new URL(url);
        if (parsedURL.origin === 'https://www.elastic.co') {
          const resp = await fetch(parsedURL);
          const data = await resp.text();
          setValue(`${initialTextValue}\n\n${data}`);
        } else {
          toasts.addWarning(_i18n.i18n.translate('console.loadFromDataUnrecognizedUrlErrorMessage', {
            defaultMessage: 'Only URLs with the Elastic domain (www.elastic.co) can be loaded in Console.'
          }));
        }
      }

      // If we have a data URI instead of HTTP, LZ-decode it. This enables
      // opening requests in Console from anywhere in Kibana.
      if (/^data:/.test(url)) {
        const data = (0, _lzString.decompressFromEncodedURIComponent)(url.replace(/^data:text\/plain,/, ''));

        // Show a toast if we have a failure
        if (data === null || data === '') {
          toasts.addWarning(_i18n.i18n.translate('console.loadFromDataUriErrorMessage', {
            defaultMessage: 'Unable to load data from the load_from query parameter in the URL'
          }));
          return;
        }
        setValue(data);
      }
    };

    // Support for loading a console snippet from a remote source, like support docs.
    const onHashChange = (0, _lodash.debounce)(async () => {
      const url = readLoadFromParam();
      if (!url) {
        return;
      }
      await loadBufferFromRemote(url);
    }, 200);
    window.addEventListener('hashchange', onHashChange);
    const loadFromParam = readLoadFromParam();
    if (loadFromParam) {
      loadBufferFromRemote(loadFromParam);
    } else {
      setValue(initialTextValue || _constants.DEFAULT_INPUT_VALUE);
    }
    return () => {
      window.removeEventListener('hashchange', onHashChange);
    };
  }, [initialTextValue, setValue, toasts]);
};
exports.useSetInitialValue = useSetInitialValue;