"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useKeyboardCommandsUtils = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _monaco = require("@kbn/monaco");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const SEND_REQUEST_ACTION_ID = 'sendRequest';
const AUTO_INDENT_ACTION_ID = 'autoIndent';
const OPEN_DOCS_ACTION_ID = 'openDocs';
const MOVE_UP_ACTION_ID = 'moveUp';
const MOVE_DOWN_ACTION_ID = 'moveDown';
const MOVE_TO_LINE_ACTION_ID = 'moveToLine';

/**
 * Hook that returns a function for registering keyboard commands in the editor.
 *
 * @param params The {@link RegisterKeyboardCommandsParams} to use.
 */
const useKeyboardCommandsUtils = () => {
  const sendRequestAction = (0, _react.useRef)(null);
  const autoIndentAction = (0, _react.useRef)(null);
  const openDocsAction = (0, _react.useRef)(null);
  const moveToPreviousAction = (0, _react.useRef)(null);
  const moveToNextAction = (0, _react.useRef)(null);
  const moveToLineAction = (0, _react.useRef)(null);
  const disposeAllActions = () => {
    if (sendRequestAction.current) {
      sendRequestAction.current.dispose();
    }
    if (autoIndentAction.current) {
      autoIndentAction.current.dispose();
    }
    if (openDocsAction.current) {
      openDocsAction.current.dispose();
    }
    if (moveToPreviousAction.current) {
      moveToPreviousAction.current.dispose();
    }
    if (moveToNextAction.current) {
      moveToNextAction.current.dispose();
    }
    if (moveToLineAction.current) {
      moveToLineAction.current.dispose();
    }
  };
  const registerKeyboardCommands = params => {
    const {
      editor,
      sendRequest,
      autoIndent,
      getDocumentationLink,
      moveToPreviousRequestEdge,
      moveToNextRequestEdge
    } = params;
    const openDocs = async () => {
      const documentation = await getDocumentationLink();
      if (!documentation) {
        return;
      }
      window.open(documentation, '_blank');
    };
    disposeAllActions();
    sendRequestAction.current = editor.addAction({
      id: SEND_REQUEST_ACTION_ID,
      label: _i18n.i18n.translate('console.keyboardCommandActionLabel.sendRequest', {
        defaultMessage: 'Send request'
      }),
      // eslint-disable-next-line no-bitwise
      keybindings: [_monaco.monaco.KeyMod.CtrlCmd | _monaco.monaco.KeyCode.Enter],
      run: sendRequest
    });
    autoIndentAction.current = editor.addAction({
      id: AUTO_INDENT_ACTION_ID,
      label: _i18n.i18n.translate('console.keyboardCommandActionLabel.autoIndent', {
        defaultMessage: 'Apply indentations'
      }),
      // eslint-disable-next-line no-bitwise
      keybindings: [_monaco.monaco.KeyMod.CtrlCmd | _monaco.monaco.KeyCode.KeyI],
      run: autoIndent
    });
    openDocsAction.current = editor.addAction({
      id: OPEN_DOCS_ACTION_ID,
      label: _i18n.i18n.translate('console.keyboardCommandActionLabel.openDocs', {
        defaultMessage: 'Open documentation'
      }),
      // eslint-disable-next-line no-bitwise
      keybindings: [_monaco.monaco.KeyMod.CtrlCmd | _monaco.monaco.KeyCode.Slash],
      run: openDocs
    });
    moveToPreviousAction.current = editor.addAction({
      id: MOVE_UP_ACTION_ID,
      label: _i18n.i18n.translate('console.keyboardCommandActionLabel.moveToPreviousRequestEdge', {
        defaultMessage: 'Move to previous request start or end'
      }),
      // eslint-disable-next-line no-bitwise
      keybindings: [_monaco.monaco.KeyMod.CtrlCmd | _monaco.monaco.KeyCode.UpArrow],
      run: moveToPreviousRequestEdge
    });
    moveToNextAction.current = editor.addAction({
      id: MOVE_DOWN_ACTION_ID,
      label: _i18n.i18n.translate('console.keyboardCommandActionLabel.moveToNextRequestEdge', {
        defaultMessage: 'Move to next request start or end'
      }),
      // eslint-disable-next-line no-bitwise
      keybindings: [_monaco.monaco.KeyMod.CtrlCmd | _monaco.monaco.KeyCode.DownArrow],
      run: moveToNextRequestEdge
    });
    moveToLineAction.current = editor.addAction({
      id: MOVE_TO_LINE_ACTION_ID,
      label: _i18n.i18n.translate('console.keyboardCommandActionLabel.moveToLine', {
        defaultMessage: 'Move cursor to a line'
      }),
      // eslint-disable-next-line no-bitwise
      keybindings: [_monaco.monaco.KeyMod.CtrlCmd | _monaco.monaco.KeyCode.KeyL],
      run: () => {
        var _prompt;
        const line = parseInt((_prompt = prompt('Enter line number')) !== null && _prompt !== void 0 ? _prompt : '', 10);
        if (!isNaN(line)) {
          editor.setPosition({
            lineNumber: line,
            column: 1
          });
        }
      }
    });
  };
  const unregisterKeyboardCommands = () => disposeAllActions();
  return {
    registerKeyboardCommands,
    unregisterKeyboardCommands
  };
};
exports.useKeyboardCommandsUtils = useKeyboardCommandsUtils;