"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createRulesWithUpdatedState = createRulesWithUpdatedState;
exports.useChangeCspRuleState = void 0;
var _public = require("@kbn/kibana-react-plugin/public");
var _reactQuery = require("@tanstack/react-query");
var _use_csp_rules_state = require("./use_csp_rules_state");
var _api = require("../../common/api");
var _use_csp_benchmark_integrations = require("../benchmarks/use_csp_benchmark_integrations");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useChangeCspRuleState = () => {
  const {
    http
  } = (0, _public.useKibana)().services;
  const queryClient = (0, _reactQuery.useQueryClient)();
  return (0, _reactQuery.useMutation)({
    mutationFn: async ruleStateUpdateRequest => {
      await (http === null || http === void 0 ? void 0 : http.post(_constants.CSP_BENCHMARK_RULES_BULK_ACTION_ROUTE_PATH, {
        version: '1',
        body: JSON.stringify({
          action: ruleStateUpdateRequest.newState,
          rules: ruleStateUpdateRequest.ruleIds
        })
      }));
    },
    onMutate: async ruleStateUpdateRequest => {
      // Cancel any outgoing refetches (so they don't overwrite our optimistic update)
      await queryClient.cancelQueries(_use_csp_rules_state.CSP_RULES_STATES_QUERY_KEY);

      // Snapshot the previous rules
      const previousCspRules = queryClient.getQueryData(_use_csp_rules_state.CSP_RULES_STATES_QUERY_KEY);

      // Optimistically update to the rules that have state changes
      queryClient.setQueryData(_use_csp_rules_state.CSP_RULES_STATES_QUERY_KEY, currentRuleStates => {
        if (!currentRuleStates) {
          return currentRuleStates;
        }
        return createRulesWithUpdatedState(ruleStateUpdateRequest, currentRuleStates);
      });

      // Return a context object with the previous value
      return {
        previousCspRules
      };
    },
    onSettled: () => {
      queryClient.invalidateQueries(_use_csp_benchmark_integrations.BENCHMARK_INTEGRATION_QUERY_KEY_V2);
      queryClient.invalidateQueries(_api.CSPM_STATS_QUERY_KEY);
      queryClient.invalidateQueries(_api.KSPM_STATS_QUERY_KEY);
      queryClient.invalidateQueries(_use_csp_rules_state.CSP_RULES_STATES_QUERY_KEY);
    },
    onError: (err, variables, context) => {
      if (context !== null && context !== void 0 && context.previousCspRules) {
        queryClient.setQueryData(_use_csp_rules_state.CSP_RULES_STATES_QUERY_KEY, context.previousCspRules);
      }
    }
  });
};
exports.useChangeCspRuleState = useChangeCspRuleState;
function createRulesWithUpdatedState(ruleStateUpdateRequest, currentRuleStates) {
  const updateRuleStates = {};
  ruleStateUpdateRequest.ruleIds.forEach(ruleId => {
    const matchingRuleKey = Object.keys(currentRuleStates).find(key => currentRuleStates[key].rule_id === ruleId.rule_id);
    if (matchingRuleKey) {
      const updatedRule = {
        ...currentRuleStates[matchingRuleKey],
        muted: ruleStateUpdateRequest.newState === 'mute'
      };
      updateRuleStates[matchingRuleKey] = updatedRule;
    }
  });
  return {
    ...currentRuleStates,
    ...updateRuleStates
  };
}