"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CSP_FLEET_PACKAGE_KUERY = exports.CLOUD_PROVIDERS = void 0;
exports.assert = assert;
exports.truthy = exports.roundScore = exports.isNonNullable = exports.isEnabledBenchmarkInputType = exports.isCspPackage = exports.getStatusForIndexName = exports.getCloudProviderNameFromAbbreviation = exports.getBenchmarkFromPackagePolicy = exports.getBenchmarkFilterQueryV2 = exports.getBenchmarkFilterQuery = exports.getBenchmarkFilter = exports.getBenchmarkCisName = exports.getBenchmarkApplicableTo = exports.extractErrorMessage = exports.cleanupCredentials = exports.calculatePostureScore = void 0;
var _common = require("@kbn/fleet-plugin/common");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * @example
 * declare const foo: Array<string | undefined | null>
 * foo.filter(isNonNullable) // foo is Array<string>
 */
const isNonNullable = v => v !== null && v !== undefined;
exports.isNonNullable = isNonNullable;
const truthy = value => !!value;
exports.truthy = truthy;
const extractErrorMessage = (e, defaultMessage = 'Unknown Error') => {
  if (e instanceof Error) return e.message;
  if (typeof e === 'string') return e;
  return defaultMessage; // TODO: i18n
};
exports.extractErrorMessage = extractErrorMessage;
const getBenchmarkFilter = (type, section) => `${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.benchmark.id: "${type}"${section ? ` AND ${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.section: "${section}"` : ''}`;
exports.getBenchmarkFilter = getBenchmarkFilter;
const isEnabledBenchmarkInputType = input => input.enabled;
exports.isEnabledBenchmarkInputType = isEnabledBenchmarkInputType;
const isCspPackage = packageName => packageName === _constants.CLOUD_SECURITY_POSTURE_PACKAGE_NAME;
exports.isCspPackage = isCspPackage;
const getBenchmarkFromPackagePolicy = inputs => {
  const enabledInputs = inputs.filter(isEnabledBenchmarkInputType);

  // Use the only enabled input
  if (enabledInputs.length === 1) {
    return getInputType(enabledInputs[0].type);
  }

  // Use the default benchmark id for multiple/none selected
  return getInputType(_constants.CLOUDBEAT_VANILLA);
};
exports.getBenchmarkFromPackagePolicy = getBenchmarkFromPackagePolicy;
const getInputType = inputType => {
  // Get the last part of the input type, input type structure: cloudbeat/<benchmark_id>
  return inputType.split('/')[1];
};
const CSP_FLEET_PACKAGE_KUERY = exports.CSP_FLEET_PACKAGE_KUERY = `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${_constants.CLOUD_SECURITY_POSTURE_PACKAGE_NAME}`;
function assert(condition, msg) {
  if (!condition) {
    throw new Error(msg);
  }
}

/**
 * @param value value is [0, 1] range
 */
const roundScore = value => Number((value * 100).toFixed(1));
exports.roundScore = roundScore;
const calculatePostureScore = (passed, failed) => {
  const total = passed + failed;
  if (total === 0) return total;
  return roundScore(passed / (passed + failed));
};
exports.calculatePostureScore = calculatePostureScore;
const getStatusForIndexName = (indexName, status) => {
  if (status) {
    const indexDetail = status.indicesDetails.find(details => details.index.indexOf(indexName) !== -1);
    if (indexDetail) {
      return indexDetail.status;
    }
  }
  return 'unknown';
};
exports.getStatusForIndexName = getStatusForIndexName;
const cleanupCredentials = packagePolicy => {
  var _enabledInput$streams, _enabledInput$streams2, _enabledInput$streams3, _enabledInput$streams4, _enabledInput$streams5, _enabledInput$streams6, _enabledInput$streams7, _enabledInput$streams8, _enabledInput$streams9;
  const enabledInput = packagePolicy.inputs.find(i => i.enabled);
  const awsCredentialType = enabledInput === null || enabledInput === void 0 ? void 0 : (_enabledInput$streams = enabledInput.streams) === null || _enabledInput$streams === void 0 ? void 0 : (_enabledInput$streams2 = _enabledInput$streams[0].vars) === null || _enabledInput$streams2 === void 0 ? void 0 : (_enabledInput$streams3 = _enabledInput$streams2['aws.credentials.type']) === null || _enabledInput$streams3 === void 0 ? void 0 : _enabledInput$streams3.value;
  const gcpCredentialType = enabledInput === null || enabledInput === void 0 ? void 0 : (_enabledInput$streams4 = enabledInput.streams) === null || _enabledInput$streams4 === void 0 ? void 0 : (_enabledInput$streams5 = _enabledInput$streams4[0].vars) === null || _enabledInput$streams5 === void 0 ? void 0 : (_enabledInput$streams6 = _enabledInput$streams5['gcp.credentials.type']) === null || _enabledInput$streams6 === void 0 ? void 0 : _enabledInput$streams6.value;
  const azureCredentialType = enabledInput === null || enabledInput === void 0 ? void 0 : (_enabledInput$streams7 = enabledInput.streams) === null || _enabledInput$streams7 === void 0 ? void 0 : (_enabledInput$streams8 = _enabledInput$streams7[0].vars) === null || _enabledInput$streams8 === void 0 ? void 0 : (_enabledInput$streams9 = _enabledInput$streams8['azure.credentials.type']) === null || _enabledInput$streams9 === void 0 ? void 0 : _enabledInput$streams9.value;
  if (awsCredentialType || gcpCredentialType || azureCredentialType) {
    let credsToKeep = [' '];
    let credFields = [' '];
    if (awsCredentialType) {
      credsToKeep = _constants.AWS_CREDENTIALS_TYPE_TO_FIELDS_MAP[awsCredentialType];
      credFields = Object.values(_constants.AWS_CREDENTIALS_TYPE_TO_FIELDS_MAP).flat();
    } else if (gcpCredentialType) {
      credsToKeep = _constants.GCP_CREDENTIALS_TYPE_TO_FIELDS_MAP[gcpCredentialType];
      credFields = Object.values(_constants.GCP_CREDENTIALS_TYPE_TO_FIELDS_MAP).flat();
    } else if (azureCredentialType) {
      credsToKeep = _constants.AZURE_CREDENTIALS_TYPE_TO_FIELDS_MAP[azureCredentialType];
      credFields = Object.values(_constants.AZURE_CREDENTIALS_TYPE_TO_FIELDS_MAP).flat();
    }
    if (credsToKeep) {
      // we need to return a copy of the policy with the unused
      // credentials set to undefined
      return {
        ...packagePolicy,
        inputs: packagePolicy.inputs.map(input => {
          if (input.enabled) {
            return {
              ...input,
              streams: input.streams.map(stream => {
                const vars = stream.vars;
                for (const field in vars) {
                  if (!credsToKeep.includes(field) && credFields.includes(field)) {
                    vars[field].value = undefined;
                  }
                }
                return {
                  ...stream,
                  vars
                };
              })
            };
          }
          return input;
        })
      };
    }
  }

  // nothing to do, return unmutated policy
  return packagePolicy;
};
exports.cleanupCredentials = cleanupCredentials;
const getBenchmarkCisName = benchmarkId => {
  switch (benchmarkId) {
    case 'cis_k8s':
      return 'CIS Kubernetes';
    case 'cis_azure':
      return 'CIS Azure';
    case 'cis_aws':
      return 'CIS AWS';
    case 'cis_eks':
      return 'CIS EKS';
    case 'cis_gcp':
      return 'CIS GCP';
  }
};
exports.getBenchmarkCisName = getBenchmarkCisName;
const CLOUD_PROVIDER_NAMES = {
  AWS: 'Amazon Web Services',
  AZURE: 'Microsoft Azure',
  GCP: 'Google Cloud Platform'
};
const CLOUD_PROVIDERS = exports.CLOUD_PROVIDERS = {
  AWS: 'aws',
  AZURE: 'azure',
  GCP: 'gcp'
};

/**
 * Returns the cloud provider name or benchmark applicable name for the given benchmark id
 */
const getBenchmarkApplicableTo = benchmarkId => {
  switch (benchmarkId) {
    case 'cis_k8s':
      return 'Kubernetes';
    case 'cis_azure':
      return CLOUD_PROVIDER_NAMES.AZURE;
    case 'cis_aws':
      return CLOUD_PROVIDER_NAMES.AWS;
    case 'cis_eks':
      return 'Amazon Elastic Kubernetes Service (EKS)';
    case 'cis_gcp':
      return CLOUD_PROVIDER_NAMES.GCP;
  }
};
exports.getBenchmarkApplicableTo = getBenchmarkApplicableTo;
const getCloudProviderNameFromAbbreviation = cloudProvider => {
  switch (cloudProvider) {
    case 'azure':
      return CLOUD_PROVIDER_NAMES.AZURE;
    case 'aws':
      return CLOUD_PROVIDER_NAMES.AWS;
    case 'gcp':
      return CLOUD_PROVIDER_NAMES.GCP;
    default:
      return cloudProvider;
  }
};
exports.getCloudProviderNameFromAbbreviation = getCloudProviderNameFromAbbreviation;
const getBenchmarkFilterQuery = (benchmarkId, benchmarkVersion, selectParams) => {
  const baseQuery = `${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.benchmark.id:${benchmarkId} AND ${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.benchmark.version:"v${benchmarkVersion}"`;
  const sectionQuery = selectParams !== null && selectParams !== void 0 && selectParams.section ? ` AND ${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.section: "${selectParams.section}"` : '';
  const ruleNumberQuery = selectParams !== null && selectParams !== void 0 && selectParams.ruleNumber ? ` AND ${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.benchmark.rule_number: "${selectParams.ruleNumber}"` : '';
  return baseQuery + sectionQuery + ruleNumberQuery;
};
exports.getBenchmarkFilterQuery = getBenchmarkFilterQuery;
const getBenchmarkFilterQueryV2 = (benchmarkId, benchmarkVersion, selectParams) => {
  const baseQuery = `${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.benchmark.id:${benchmarkId} AND ${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.benchmark.version:"v${benchmarkVersion}"`;
  let sectionQuery = '';
  let ruleNumberQuery = '';
  if (selectParams !== null && selectParams !== void 0 && selectParams.section) {
    var _selectParams$section;
    const sectionParamsArr = (_selectParams$section = selectParams.section) === null || _selectParams$section === void 0 ? void 0 : _selectParams$section.map(params => `${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.section: "${params}"`);
    sectionQuery = ' AND (' + sectionParamsArr.join(' OR ') + ')';
  }
  if (selectParams !== null && selectParams !== void 0 && selectParams.ruleNumber) {
    var _selectParams$ruleNum;
    const ruleNumbersParamsArr = (_selectParams$ruleNum = selectParams.ruleNumber) === null || _selectParams$ruleNum === void 0 ? void 0 : _selectParams$ruleNum.map(params => `${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.benchmark.rule_number: "${params}"`);
    ruleNumberQuery = ' AND (' + ruleNumbersParamsArr.join(' OR ') + ')';
  }
  return baseQuery + sectionQuery + ruleNumberQuery;
};
exports.getBenchmarkFilterQueryV2 = getBenchmarkFilterQueryV2;