"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.create = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _server = require("@kbn/core/server");
var _domain = require("../../../common/types/domain");
var _runtime_types = require("../../common/runtime_types");
var _authorization = require("../../authorization");
var _error = require("../../common/error");
var _utils = require("../../common/utils");
var _constants = require("../../common/constants");
var _api = require("../../../common/types/api");
require("../utils");
var _validators = require("./validators");
var _utils3 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Creates a new case.
 *
 */
const create = async (data, clientArgs, casesClient) => {
  const {
    services: {
      caseService,
      userActionService,
      licensingService,
      notificationService
    },
    user,
    logger,
    authorization: auth
  } = clientArgs;
  try {
    var _configurations$, _query$severity, _query$assignees, _query$category, _query$customFields;
    const query = (0, _runtime_types.decodeWithExcessOrThrow)(_api.CasePostRequestRt)(data);
    const configurations = await casesClient.configure.get({
      owner: data.owner
    });
    const customFieldsConfiguration = (_configurations$ = configurations[0]) === null || _configurations$ === void 0 ? void 0 : _configurations$.customFields;
    const customFieldsValidationParams = {
      requestCustomFields: data.customFields,
      customFieldsConfiguration
    };
    (0, _validators.validateCustomFields)(customFieldsValidationParams);
    const savedObjectID = _server.SavedObjectsUtils.generateId();
    await auth.ensureAuthorized({
      operation: _authorization.Operations.createCase,
      entities: [{
        owner: query.owner,
        id: savedObjectID
      }]
    });

    /**
     * Assign users to a case is only available to Platinum+
     */

    if (query.assignees && query.assignees.length !== 0) {
      const hasPlatinumLicenseOrGreater = await licensingService.isAtLeastPlatinum();
      if (!hasPlatinumLicenseOrGreater) {
        throw _boom.default.forbidden('In order to assign users to cases, you must be subscribed to an Elastic Platinum license');
      }
      licensingService.notifyUsage(_constants.LICENSING_CASE_ASSIGNMENT_FEATURE);
    }

    /**
     * Trim title, category, description and tags
     * and fill out missing custom fields
     * before saving to ES
     */

    const normalizedCase = (0, _utils3.normalizeCreateCaseRequest)(query, customFieldsConfiguration);
    const newCase = await caseService.createCase({
      attributes: (0, _utils.transformNewCase)({
        user,
        newCase: normalizedCase
      }),
      id: savedObjectID,
      refresh: false
    });
    await userActionService.creator.createUserAction({
      userAction: {
        type: _domain.UserActionTypes.create_case,
        caseId: newCase.id,
        user,
        payload: {
          ...query,
          severity: (_query$severity = query.severity) !== null && _query$severity !== void 0 ? _query$severity : _domain.CaseSeverity.LOW,
          assignees: (_query$assignees = query.assignees) !== null && _query$assignees !== void 0 ? _query$assignees : [],
          category: (_query$category = query.category) !== null && _query$category !== void 0 ? _query$category : null,
          customFields: (_query$customFields = query.customFields) !== null && _query$customFields !== void 0 ? _query$customFields : []
        },
        owner: newCase.attributes.owner
      }
    });
    if (query.assignees && query.assignees.length !== 0) {
      const assigneesWithoutCurrentUser = query.assignees.filter(assignee => assignee.uid !== user.profile_uid);
      await notificationService.notifyAssignees({
        assignees: assigneesWithoutCurrentUser,
        theCase: newCase
      });
    }
    const res = (0, _utils.flattenCaseSavedObject)({
      savedObject: newCase
    });
    return (0, _runtime_types.decodeOrThrow)(_domain.CaseRt)(res);
  } catch (error) {
    throw (0, _error.createCaseError)({
      message: `Failed to create case: ${error}`,
      error,
      logger
    });
  }
};
exports.create = create;