"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseURL = exports.parseCaseUsers = exports.isEmptyValue = exports.isDeprecatedConnector = exports.getConnectorsFormValidators = exports.getConnectorsFormSerializer = exports.getConnectorsFormDeserializer = exports.getConnectorIcon = exports.getConnectorById = exports.customFieldsFormSerializer = exports.customFieldsFormDeserializer = exports.convertEmptyValuesToNull = exports.convertCustomFieldValue = exports.connectorDeprecationValidator = exports.addOrReplaceField = void 0;
exports.removeEmptyFields = removeEmptyFields;
exports.stringifyToURL = exports.removeItemFromSessionStorage = void 0;
var _lodash = require("lodash");
var _domain = require("../../common/types/domain");
var _validator = require("./connectors/swimlane/validator");
var _user_converter = require("./user_profiles/user_converter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getConnectorById = (id, connectors) => {
  var _connectors$find;
  return (_connectors$find = connectors.find(c => c.id === id)) !== null && _connectors$find !== void 0 ? _connectors$find : null;
};
exports.getConnectorById = getConnectorById;
const validators = {
  [_domain.ConnectorTypes.swimlane]: _validator.connectorValidator
};
const connectorDeprecationValidator = connector => {
  if (connector.isDeprecated) {
    return {
      message: 'Deprecated connector'
    };
  }
};
exports.connectorDeprecationValidator = connectorDeprecationValidator;
const getConnectorsFormValidators = ({
  connectors = [],
  config = {}
}) => ({
  ...config,
  validations: [{
    validator: ({
      value: connectorId
    }) => {
      const connector = getConnectorById(connectorId, connectors);
      if (connector != null) {
        return connectorDeprecationValidator(connector);
      }
    }
  }, {
    validator: ({
      value: connectorId
    }) => {
      const connector = getConnectorById(connectorId, connectors);
      if (connector != null) {
        var _validators$connector;
        return (_validators$connector = validators[connector.actionTypeId]) === null || _validators$connector === void 0 ? void 0 : _validators$connector.call(validators, connector);
      }
    }
  }]
});

/**
 * Fields without a value need to be transformed to null.
 * Passing undefined for a field to the backed will throw an error.
 * Fo that reason, we need to convert empty fields to null.
 */
exports.getConnectorsFormValidators = getConnectorsFormValidators;
const getConnectorsFormSerializer = data => {
  if (data.fields) {
    const serializedFields = convertEmptyValuesToNull(data.fields);
    return {
      ...data,
      fields: serializedFields
    };
  }
  return data;
};
exports.getConnectorsFormSerializer = getConnectorsFormSerializer;
const convertEmptyValuesToNull = fields => {
  if (fields) {
    return Object.entries(fields).reduce((acc, [key, value]) => {
      return {
        ...acc,
        [key]: isEmptyValue(value) ? null : value
      };
    }, {});
  }
  return null;
};

/**
 * We cannot use lodash isEmpty util function
 * because it will return true for primitive values
 * like boolean or numbers
 */
exports.convertEmptyValuesToNull = convertEmptyValuesToNull;
const isEmptyValue = value => value === null || value === undefined || typeof value === 'object' && Object.keys(value).length === 0 || typeof value === 'string' && value.trim().length === 0;

/**
 * Form html elements do not support null values.
 * For that reason, we need to convert null values to
 * undefined which is supported.
 */
exports.isEmptyValue = isEmptyValue;
const getConnectorsFormDeserializer = data => {
  if (data.fields) {
    const deserializedFields = Object.entries(data.fields).reduce((acc, [key, value]) => ({
      ...acc,
      [key]: value === null ? undefined : value
    }), {});
    return {
      ...data,
      fields: deserializedFields
    };
  }
  return data;
};
exports.getConnectorsFormDeserializer = getConnectorsFormDeserializer;
const getConnectorIcon = (triggersActionsUi, type) => {
  /**
   * triggersActionsUi.actionTypeRegistry.get will throw an error if the type is not registered.
   * This will break Kibana if not handled properly.
   */
  const emptyResponse = '';
  if (type == null) {
    return emptyResponse;
  }
  try {
    if (triggersActionsUi.actionTypeRegistry.has(type)) {
      return triggersActionsUi.actionTypeRegistry.get(type).iconClass;
    }
  } catch {
    return emptyResponse;
  }
  return emptyResponse;
};
exports.getConnectorIcon = getConnectorIcon;
const isDeprecatedConnector = connector => {
  var _connector$isDeprecat;
  return (_connector$isDeprecat = connector === null || connector === void 0 ? void 0 : connector.isDeprecated) !== null && _connector$isDeprecat !== void 0 ? _connector$isDeprecat : false;
};
exports.isDeprecatedConnector = isDeprecatedConnector;
const removeItemFromSessionStorage = key => {
  window.sessionStorage.removeItem(key);
};
exports.removeItemFromSessionStorage = removeItemFromSessionStorage;
const stringifyToURL = parsedParams => new URLSearchParams(parsedParams).toString();
exports.stringifyToURL = stringifyToURL;
const parseURL = queryString => Object.fromEntries(new URLSearchParams(queryString));
exports.parseURL = parseURL;
const parseCaseUsers = ({
  caseUsers,
  createdBy
}) => {
  const userProfiles = new Map();
  const reporterAsArray = (caseUsers === null || caseUsers === void 0 ? void 0 : caseUsers.reporter) != null ? [caseUsers.reporter] : [(0, _user_converter.convertToCaseUserWithProfileInfo)(createdBy)];
  if (caseUsers) {
    for (const user of [...caseUsers.assignees, ...caseUsers.participants, caseUsers.reporter, ...caseUsers.unassignedUsers]) {
      /**
       * If the user has a valid profile UID and a valid username
       * then the backend successfully fetched the user profile
       * information from the security plugin. Checking only for the
       * profile UID is not enough as a user can use our API to add
       * an assignee with a non existing UID.
       */
      if (user.uid != null && user.user.username != null) {
        userProfiles.set(user.uid, {
          uid: user.uid,
          user: user.user,
          data: {
            avatar: user.avatar
          }
        });
      }
    }
  }
  return {
    userProfiles,
    reporterAsArray
  };
};
exports.parseCaseUsers = parseCaseUsers;
const convertCustomFieldValue = value => {
  if (typeof value === 'string' && (0, _lodash.isEmpty)(value)) {
    return null;
  }
  return value;
};
exports.convertCustomFieldValue = convertCustomFieldValue;
const addOrReplaceField = (fields, fieldToAdd) => {
  const foundFieldIndex = fields.findIndex(field => field.key === fieldToAdd.key);
  if (foundFieldIndex === -1) {
    return [...fields, fieldToAdd];
  }
  return fields.map(field => {
    if (field.key !== fieldToAdd.key) {
      return field;
    }
    return fieldToAdd;
  });
};
exports.addOrReplaceField = addOrReplaceField;
function removeEmptyFields(obj) {
  return Object.fromEntries(Object.entries(obj).filter(([_, value]) => !(0, _lodash.isEmpty)(value) || typeof value === 'boolean').map(([key, value]) => [key, value === Object(value) && !Array.isArray(value) ? removeEmptyFields(value) : value]));
}
const customFieldsFormDeserializer = customFields => {
  if (!customFields || !customFields.length) {
    return null;
  }
  return customFields.reduce((acc, customField) => {
    const initial = {
      [customField.key]: customField.value
    };
    return {
      ...acc,
      ...initial
    };
  }, {});
};
exports.customFieldsFormDeserializer = customFieldsFormDeserializer;
const customFieldsFormSerializer = (customFields, selectedCustomFieldsConfiguration) => {
  const transformedCustomFields = [];
  if (!customFields || !selectedCustomFieldsConfiguration.length) {
    return [];
  }
  for (const [key, value] of Object.entries(customFields)) {
    const configCustomField = selectedCustomFieldsConfiguration.find(item => item.key === key);
    if (configCustomField) {
      transformedCustomFields.push({
        key: configCustomField.key,
        type: configCustomField.type,
        value: convertCustomFieldValue(value)
      });
    }
  }
  return transformedCustomFields;
};
exports.customFieldsFormSerializer = customFieldsFormSerializer;