"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createOrUpdateStaticDataView = createOrUpdateStaticDataView;
var _server = require("@kbn/core/server");
var _i18n = require("@kbn/i18n");
var _apmDataView = require("@kbn/apm-data-view");
var _apm = require("../../../common/es_fields/apm");
var _has_historical_agent_data = require("../historical_data/has_historical_agent_data");
var _with_apm_span = require("../../utils/with_apm_span");
var _get_apm_data_view_index_pattern = require("./get_apm_data_view_index_pattern");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function createOrUpdateStaticDataView({
  dataViewService,
  resources,
  apmEventClient,
  spaceId,
  logger
}) {
  const {
    config
  } = resources;
  const dataViewId = (0, _apmDataView.getStaticDataViewId)(spaceId);
  logger.info(`create static data view ${dataViewId}`);
  return (0, _with_apm_span.withApmSpan)('create_static_data_view', async () => {
    // don't auto-create APM data view if it's been disabled via the config
    if (!config.autoCreateApmDataView) {
      return {
        created: false,
        reason: _i18n.i18n.translate('xpack.apm.dataView.autoCreateDisabled', {
          defaultMessage: 'Auto-creation of data views has been disabled via "autoCreateApmDataView" config option'
        })
      };
    }

    // Discover and other apps will throw errors if an data view exists without having matching indices.
    // The following ensures the data view is only created if APM data is found
    const hasData = await (0, _has_historical_agent_data.hasHistoricalAgentData)(apmEventClient);
    if (!hasData) {
      return {
        created: false,
        reason: _i18n.i18n.translate('xpack.apm.dataView.noApmData', {
          defaultMessage: 'No APM data'
        })
      };
    }
    const apmDataViewIndexPattern = (0, _get_apm_data_view_index_pattern.getApmDataViewIndexPattern)(apmEventClient.indices);
    const shouldCreateOrUpdate = await getShouldCreateOrUpdate({
      apmDataViewIndexPattern,
      dataViewService,
      dataViewId
    });
    if (!shouldCreateOrUpdate) {
      return {
        created: false,
        reason: _i18n.i18n.translate('xpack.apm.dataView.alreadyExistsInActiveSpace', {
          defaultMessage: 'Dataview already exists in the active space and does not need to be updated'
        })
      };
    }

    // delete legacy global data view

    const dataView = await createAndSaveStaticDataView({
      dataViewService,
      apmDataViewIndexPattern,
      dataViewId
    });
    return {
      created: true,
      dataView
    };
  });
}

// only create data view if it doesn't exist or was changed
async function getShouldCreateOrUpdate({
  dataViewService,
  apmDataViewIndexPattern,
  dataViewId
}) {
  try {
    const existingDataView = await dataViewService.get(dataViewId);
    return existingDataView.getIndexPattern() !== apmDataViewIndexPattern;
  } catch (e) {
    // ignore exception if the data view (saved object) is not found
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
      return true;
    }
    throw e;
  }
}
function createAndSaveStaticDataView({
  dataViewService,
  apmDataViewIndexPattern,
  dataViewId
}) {
  return dataViewService.createAndSave({
    allowNoIndex: true,
    id: dataViewId,
    name: 'APM',
    title: apmDataViewIndexPattern,
    timeFieldName: '@timestamp',
    // link to APM from Discover
    fieldFormats: {
      [_apm.TRACE_ID]: {
        id: 'url',
        params: {
          urlTemplate: 'apm/link-to/trace/{{value}}',
          labelTemplate: '{{value}}'
        }
      },
      [_apm.TRANSACTION_ID]: {
        id: 'url',
        params: {
          urlTemplate: 'apm/link-to/transaction/{{value}}',
          labelTemplate: '{{value}}'
        }
      },
      [_apm.TRANSACTION_DURATION]: {
        id: 'duration',
        params: {
          inputFormat: 'microseconds',
          outputFormat: 'asMilliseconds',
          showSuffix: true,
          useShortSuffix: true,
          outputPrecision: 2,
          includeSpaceWithSuffix: true
        }
      }
    }
  }, true);
}