"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAlertDetailsContextHandler = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _lodash = require("lodash");
var _apm = require("../../../../common/es_fields/apm");
var _get_apm_alerts_client = require("../../../lib/helpers/get_apm_alerts_client");
var _get_apm_event_client = require("../../../lib/helpers/get_apm_event_client");
var _get_ml_client = require("../../../lib/helpers/get_ml_client");
var _get_random_sampler = require("../../../lib/helpers/get_random_sampler");
var _get_apm_service_summary = require("../get_apm_service_summary");
var _get_apm_downstream_dependencies = require("../get_apm_downstream_dependencies");
var _get_log_categories = require("../get_log_categories");
var _get_anomalies = require("../get_apm_service_summary/get_anomalies");
var _get_service_name_from_signals = require("./get_service_name_from_signals");
var _get_container_id_from_signals = require("./get_container_id_from_signals");
var _get_changepoints = require("../get_changepoints");
var _get_apm_errors = require("./get_apm_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAlertDetailsContextHandler = (resourcePlugins, logger) => {
  return async (requestContext, query) => {
    const resources = {
      getApmIndices: async () => {
        const coreContext = await requestContext.core;
        return resourcePlugins.apmDataAccess.setup.getApmIndices(coreContext.savedObjects.client);
      },
      request: requestContext.request,
      params: {
        query: {
          _inspect: false
        }
      },
      plugins: resourcePlugins,
      context: {
        core: requestContext.core,
        licensing: requestContext.licensing,
        alerting: resourcePlugins.alerting.start().then(startContract => {
          return {
            getRulesClient() {
              return startContract.getRulesClientWithRequest(requestContext.request);
            }
          };
        }),
        rac: resourcePlugins.ruleRegistry.start().then(startContract => {
          return {
            getAlertsClient() {
              return startContract.getRacClientWithRequest(requestContext.request);
            }
          };
        })
      }
    };
    const [apmEventClient, annotationsClient, apmAlertsClient, coreContext, mlClient, randomSampler] = await Promise.all([(0, _get_apm_event_client.getApmEventClient)(resources), resourcePlugins.observability.setup.getScopedAnnotationsClient(resources.context, requestContext.request), (0, _get_apm_alerts_client.getApmAlertsClient)(resources), requestContext.core, (0, _get_ml_client.getMlClient)(resources), (0, _get_random_sampler.getRandomSampler)({
      security: resourcePlugins.security,
      probability: 1,
      request: requestContext.request
    })]);
    const esClient = coreContext.elasticsearch.client.asCurrentUser;
    const alertStartedAt = query.alert_started_at;
    const serviceEnvironment = query['service.environment'];
    const hostName = query['host.name'];
    const kubernetesPodName = query['kubernetes.pod.name'];
    const [serviceName, containerId] = await Promise.all([(0, _get_service_name_from_signals.getServiceNameFromSignals)({
      query,
      esClient,
      coreContext,
      apmEventClient
    }), (0, _get_container_id_from_signals.getContainerIdFromSignals)({
      query,
      esClient,
      coreContext,
      apmEventClient
    })]);
    const downstreamDependenciesPromise = serviceName ? (0, _get_apm_downstream_dependencies.getAssistantDownstreamDependencies)({
      apmEventClient,
      arguments: {
        serviceName,
        serviceEnvironment,
        start: (0, _moment.default)(alertStartedAt).subtract(24, 'hours').toISOString(),
        end: alertStartedAt
      },
      randomSampler
    }) : undefined;
    const dataFetchers = [];

    // service summary
    if (serviceName) {
      dataFetchers.push(async () => {
        const serviceSummary = await (0, _get_apm_service_summary.getApmServiceSummary)({
          apmEventClient,
          annotationsClient,
          esClient,
          apmAlertsClient,
          mlClient,
          logger,
          arguments: {
            'service.name': serviceName,
            'service.environment': serviceEnvironment,
            start: (0, _moment.default)(alertStartedAt).subtract(5, 'minute').toISOString(),
            end: alertStartedAt
          }
        });
        return {
          key: 'serviceSummary',
          description: `Metadata for the service "${serviceName}" that produced the alert. The alert might be caused by an issue in the service itself or one of its dependencies.`,
          data: serviceSummary
        };
      });
    }

    // downstream dependencies
    if (serviceName) {
      dataFetchers.push(async () => {
        const downstreamDependencies = await downstreamDependenciesPromise;
        return {
          key: 'downstreamDependencies',
          description: `Downstream dependencies from the service "${serviceName}". Problems in these services can negatively affect the performance of "${serviceName}"`,
          data: downstreamDependencies
        };
      });
    }

    // log categories
    dataFetchers.push(async () => {
      const downstreamDependencies = await downstreamDependenciesPromise;
      const {
        logCategories,
        entities
      } = await (0, _get_log_categories.getLogCategories)({
        apmEventClient,
        esClient,
        coreContext,
        arguments: {
          start: (0, _moment.default)(alertStartedAt).subtract(15, 'minute').toISOString(),
          end: alertStartedAt,
          entities: {
            'service.name': serviceName,
            'host.name': hostName,
            'container.id': containerId,
            'kubernetes.pod.name': kubernetesPodName
          }
        }
      });
      const entitiesAsString = entities.map(({
        key,
        value
      }) => `${key}:${value}`).join(', ');
      return {
        key: 'logCategories',
        description: `Log events occurring up to 15 minutes before the alert was triggered. Filtered by the entities: ${entitiesAsString}`,
        data: logCategoriesWithDownstreamServiceName(logCategories, downstreamDependencies)
      };
    });

    // apm errors
    if (serviceName) {
      dataFetchers.push(async () => {
        const apmErrors = await (0, _get_apm_errors.getApmErrors)({
          apmEventClient,
          start: (0, _moment.default)(alertStartedAt).subtract(15, 'minute').toISOString(),
          end: alertStartedAt,
          serviceName,
          serviceEnvironment
        });
        const downstreamDependencies = await downstreamDependenciesPromise;
        const errorsWithDownstreamServiceName = getApmErrorsWithDownstreamServiceName(apmErrors, downstreamDependencies);
        return {
          key: 'apmErrors',
          description: `Exceptions (errors) thrown by the service "${serviceName}". If an error contains a downstream service name this could be a possible root cause. If relevant please describe what the error means and what it could be caused by.`,
          data: errorsWithDownstreamServiceName
        };
      });
    }

    // exit span change points
    dataFetchers.push(async () => {
      const exitSpanChangePoints = await (0, _get_changepoints.getExitSpanChangePoints)({
        apmEventClient,
        start: (0, _moment.default)(alertStartedAt).subtract(6, 'hours').toISOString(),
        end: alertStartedAt,
        serviceName,
        serviceEnvironment
      });
      return {
        key: 'exitSpanChangePoints',
        description: `Significant change points for the dependencies of "${serviceName}". Use this to spot dips or spikes in throughput, latency and failure rate for downstream dependencies`,
        data: exitSpanChangePoints
      };
    });

    // service change points
    dataFetchers.push(async () => {
      const serviceChangePoints = await (0, _get_changepoints.getServiceChangePoints)({
        apmEventClient,
        start: (0, _moment.default)(alertStartedAt).subtract(6, 'hours').toISOString(),
        end: alertStartedAt,
        serviceName,
        serviceEnvironment,
        transactionType: query['transaction.type'],
        transactionName: query['transaction.name']
      });
      return {
        key: 'serviceChangePoints',
        description: `Significant change points for "${serviceName}". Use this to spot dips and spikes in throughput, latency and failure rate`,
        data: serviceChangePoints
      };
    });

    // Anomalies
    dataFetchers.push(async () => {
      const anomalies = await (0, _get_anomalies.getAnomalies)({
        start: (0, _moment.default)(alertStartedAt).subtract(1, 'hour').valueOf(),
        end: (0, _moment.default)(alertStartedAt).valueOf(),
        environment: serviceEnvironment,
        mlClient,
        logger
      });
      return {
        key: 'anomalies',
        description: `Anomalies for services running in the environment "${serviceEnvironment}". Anomalies are detected using machine learning and can help you spot unusual patterns in your data.`,
        data: anomalies
      };
    });
    const items = await Promise.all(dataFetchers.map(async dataFetcher => {
      try {
        return await dataFetcher();
      } catch (error) {
        logger.error('Error while fetching observability alert details context');
        logger.error(error);
        return;
      }
    }));
    return items.filter(item => item && !(0, _lodash.isEmpty)(item.data));
  };
};
exports.getAlertDetailsContextHandler = getAlertDetailsContextHandler;
function getApmErrorsWithDownstreamServiceName(apmErrors, downstreamDependencies) {
  return apmErrors === null || apmErrors === void 0 ? void 0 : apmErrors.map(({
    name,
    lastSeen,
    occurrences,
    downstreamServiceResource
  }) => {
    var _downstreamDependenci;
    const downstreamServiceName = downstreamDependencies === null || downstreamDependencies === void 0 ? void 0 : (_downstreamDependenci = downstreamDependencies.find(dependency => dependency['span.destination.service.resource'] === downstreamServiceResource)) === null || _downstreamDependenci === void 0 ? void 0 : _downstreamDependenci['service.name'];
    return {
      message: name,
      lastSeen: new Date(lastSeen).toISOString(),
      occurrences,
      downstream: {
        [_apm.SPAN_DESTINATION_SERVICE_RESOURCE]: downstreamServiceResource,
        [_apm.SERVICE_NAME]: downstreamServiceName
      }
    };
  });
}
function logCategoriesWithDownstreamServiceName(logCategories, downstreamDependencies) {
  return logCategories === null || logCategories === void 0 ? void 0 : logCategories.map(({
    errorCategory,
    docCount,
    sampleMessage,
    downstreamServiceResource
  }) => {
    var _downstreamDependenci2;
    const downstreamServiceName = downstreamDependencies === null || downstreamDependencies === void 0 ? void 0 : (_downstreamDependenci2 = downstreamDependencies.find(dependency => dependency['span.destination.service.resource'] === downstreamServiceResource)) === null || _downstreamDependenci2 === void 0 ? void 0 : _downstreamDependenci2['service.name'];
    return {
      errorCategory,
      docCount,
      sampleMessage,
      downstream: {
        [_apm.SPAN_DESTINATION_SERVICE_RESOURCE]: downstreamServiceResource,
        [_apm.SERVICE_NAME]: downstreamServiceName
      }
    };
  });
}