"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAnomalies = getAnomalies;
var _server = require("@kbn/observability-plugin/server");
var _lodash = require("lodash");
var _apm = require("../../../../common/es_fields/apm");
var _apm_ml_detectors = require("../../../../common/anomaly_detection/apm_ml_detectors");
var _as_mutable_array = require("../../../../common/utils/as_mutable_array");
var _maybe = require("../../../../common/utils/maybe");
var _anomaly_search = require("../../../lib/anomaly_detection/anomaly_search");
var _apm_ml_anomaly_query = require("../../../lib/anomaly_detection/apm_ml_anomaly_query");
var _apm_ml_jobs_query = require("../../../lib/anomaly_detection/apm_ml_jobs_query");
var _get_ml_jobs_with_apm_group = require("../../../lib/anomaly_detection/get_ml_jobs_with_apm_group");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getAnomalies({
  serviceName,
  transactionType,
  environment,
  start,
  end,
  mlClient,
  logger
}) {
  var _anomaliesResponse$ag;
  if (!mlClient) {
    return [];
  }
  const mlJobs = (await (0, _get_ml_jobs_with_apm_group.getMlJobsWithAPMGroup)(mlClient.anomalyDetectors)).filter(job => job.environment === environment);
  if (!mlJobs.length) {
    return [];
  }
  const anomaliesResponse = await (0, _anomaly_search.anomalySearch)(mlClient.mlSystem.mlAnomalySearch, {
    body: {
      size: 0,
      query: {
        bool: {
          filter: [...(0, _apm_ml_anomaly_query.apmMlAnomalyQuery)({
            serviceName,
            transactionType
          }), ...(0, _server.rangeQuery)(start, end, 'timestamp'), ...(0, _apm_ml_jobs_query.apmMlJobsQuery)(mlJobs)]
        }
      },
      aggs: {
        by_timeseries_id: {
          composite: {
            size: 5000,
            sources: (0, _as_mutable_array.asMutableArray)([{
              jobId: {
                terms: {
                  field: 'job_id'
                }
              }
            }, {
              detectorIndex: {
                terms: {
                  field: 'detector_index'
                }
              }
            }, {
              serviceName: {
                terms: {
                  field: 'partition_field_value'
                }
              }
            }, {
              transactionType: {
                terms: {
                  field: 'by_field_value'
                }
              }
            }])
          },
          aggs: {
            record_scores: {
              filter: {
                term: {
                  result_type: 'record'
                }
              },
              aggs: {
                top_anomaly: {
                  top_metrics: {
                    metrics: (0, _as_mutable_array.asMutableArray)([{
                      field: 'record_score'
                    }, {
                      field: 'actual'
                    }, {
                      field: 'timestamp'
                    }]),
                    size: 1,
                    sort: {
                      record_score: 'desc'
                    }
                  }
                }
              }
            },
            model_lower: {
              min: {
                field: 'model_lower'
              }
            },
            model_upper: {
              max: {
                field: 'model_upper'
              }
            }
          }
        }
      }
    }
  });
  const jobsById = (0, _lodash.keyBy)(mlJobs, job => job.jobId);
  const anomalies = (_anomaliesResponse$ag = anomaliesResponse.aggregations) === null || _anomaliesResponse$ag === void 0 ? void 0 : _anomaliesResponse$ag.by_timeseries_id.buckets.map(bucket => {
    var _bucket$record_scores;
    const jobId = bucket.key.jobId;
    const job = (0, _maybe.maybe)(jobsById[jobId]);
    if (!job) {
      logger.warn(`Could not find job for id ${jobId}`);
      return undefined;
    }
    const type = (0, _apm_ml_detectors.getAnomalyDetectorType)(Number(bucket.key.detectorIndex));

    // ml failure rate is stored as 0-100, we calculate failure rate as 0-1
    const divider = type === _apm_ml_detectors.AnomalyDetectorType.txFailureRate ? 100 : 1;
    const metrics = (_bucket$record_scores = bucket.record_scores.top_anomaly.top[0]) === null || _bucket$record_scores === void 0 ? void 0 : _bucket$record_scores.metrics;
    if (!metrics) {
      return undefined;
    }
    return {
      '@timestamp': new Date(metrics.timestamp).toISOString(),
      metricName: type.replace('tx', 'transaction'),
      [_apm.SERVICE_NAME]: bucket.key.serviceName,
      [_apm.SERVICE_ENVIRONMENT]: job.environment,
      [_apm.TRANSACTION_TYPE]: bucket.key.transactionType,
      anomalyScore: metrics.record_score,
      actualValue: Number(metrics.actual) / divider,
      expectedBoundsLower: Number(bucket.model_lower.value) / divider,
      expectedBoundsUpper: Number(bucket.model_upper.value) / divider
    };
  });
  return (0, _lodash.compact)(anomalies);
}