"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAnomalyRuleType = registerAnomalyRuleType;
var _server = require("@kbn/alerting-plugin/server");
var _server2 = require("@kbn/core/server");
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _common = require("@kbn/observability-plugin/common");
var _server3 = require("@kbn/observability-plugin/server");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _common2 = require("@kbn/spaces-plugin/common");
var _std = require("@kbn/std");
var _lodash = require("lodash");
var _anomaly_detection = require("../../../../../common/anomaly_detection");
var _apm = require("../../../../../common/es_fields/apm");
var _environment_filter_values = require("../../../../../common/environment_filter_values");
var _apm_rule_types = require("../../../../../common/rules/apm_rule_types");
var _as_mutable_array = require("../../../../../common/utils/as_mutable_array");
var _formatters = require("../../../../../common/utils/formatters");
var _get_service_anomalies = require("../../../service_map/get_service_anomalies");
var _action_variables = require("../../action_variables");
var _register_apm_rule_types = require("../../register_apm_rule_types");
var _get_service_group_fields_for_anomaly = require("./get_service_group_fields_for_anomaly");
var _schema = require("../../../../../common/rules/schema");
var _apm_ml_detectors = require("../../../../../common/anomaly_detection/apm_ml_detectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ruleTypeConfig = _apm_rule_types.RULE_TYPES_CONFIG[_ruleDataUtils.ApmRuleType.Anomaly];
function registerAnomalyRuleType({
  alerting,
  alertsLocator,
  getApmIndices,
  basePath,
  logger,
  ml,
  ruleDataClient
}) {
  if (!alerting) {
    throw new Error('Cannot register anomaly rule type. The alerting plugin needs to be enabled.');
  }
  alerting.registerType({
    id: _ruleDataUtils.ApmRuleType.Anomaly,
    name: ruleTypeConfig.name,
    actionGroups: ruleTypeConfig.actionGroups,
    defaultActionGroupId: ruleTypeConfig.defaultActionGroupId,
    validate: {
      params: _schema.anomalyParamsSchema
    },
    schemas: {
      params: {
        type: 'config-schema',
        schema: _schema.anomalyParamsSchema
      }
    },
    actionVariables: {
      context: [_action_variables.apmActionVariables.alertDetailsUrl, _action_variables.apmActionVariables.environment, _action_variables.apmActionVariables.reason, _action_variables.apmActionVariables.serviceName, _action_variables.apmActionVariables.threshold, _action_variables.apmActionVariables.transactionType, _action_variables.apmActionVariables.triggerValue, _action_variables.apmActionVariables.viewInAppUrl]
    },
    category: _server2.DEFAULT_APP_CATEGORIES.observability.id,
    producer: _apm_rule_types.APM_SERVER_FEATURE_ID,
    minimumLicenseRequired: 'basic',
    isExportable: true,
    executor: async options => {
      var _ruleParams$anomalyDe, _ruleParams$anomalyDe2, _response$aggregation, _response$aggregation2;
      if (!ml) {
        return {
          state: {}
        };
      }
      const {
        params,
        services,
        spaceId,
        startedAt,
        getTimeRange
      } = options;
      const {
        alertsClient,
        savedObjectsClient,
        scopedClusterClient
      } = services;
      if (!alertsClient) {
        throw new _server.AlertsClientError();
      }
      const apmIndices = await getApmIndices(savedObjectsClient);
      const ruleParams = params;
      const request = {};
      const {
        mlAnomalySearch
      } = ml.mlSystemProvider(request, savedObjectsClient);
      const anomalyDetectors = ml.anomalyDetectorsProvider(request, savedObjectsClient);
      const mlJobs = await (0, _get_service_anomalies.getMLJobs)(anomalyDetectors, ruleParams.environment);
      const selectedOption = _apm_rule_types.ANOMALY_ALERT_SEVERITY_TYPES.find(option => option.type === ruleParams.anomalySeverityType);
      if (!selectedOption) {
        throw new Error(`Anomaly alert severity type ${ruleParams.anomalySeverityType} is not supported.`);
      }
      const threshold = selectedOption.threshold;
      if (mlJobs.length === 0) {
        return {
          state: {}
        };
      }

      // Lookback window must be at least 30m to support rules created before this change where default was 15m
      const minimumWindow = '30m';
      const requestedWindow = `${ruleParams.windowSize}${ruleParams.windowUnit}`;
      const window = _datemath.default.parse(`now-${minimumWindow}`).valueOf() < _datemath.default.parse(`now-${requestedWindow}`).valueOf() ? minimumWindow : requestedWindow;
      const {
        dateStart
      } = getTimeRange(window);
      const jobIds = mlJobs.map(job => job.jobId);
      const anomalySearchParams = {
        body: {
          track_total_hits: false,
          size: 0,
          query: {
            bool: {
              filter: [{
                term: {
                  result_type: 'record'
                }
              }, {
                terms: {
                  job_id: jobIds
                }
              }, {
                term: {
                  is_interim: false
                }
              }, {
                range: {
                  timestamp: {
                    gte: dateStart
                  }
                }
              }, ...(0, _server3.termQuery)('partition_field_value', ruleParams.serviceName, {
                queryEmptyString: false
              }), ...(0, _server3.termQuery)('by_field_value', ruleParams.transactionType, {
                queryEmptyString: false
              }), ...(0, _server3.termsQuery)('detector_index', ...((_ruleParams$anomalyDe = (_ruleParams$anomalyDe2 = ruleParams.anomalyDetectorTypes) === null || _ruleParams$anomalyDe2 === void 0 ? void 0 : _ruleParams$anomalyDe2.map(type => (0, _apm_ml_detectors.getAnomalyDetectorIndex)(type))) !== null && _ruleParams$anomalyDe !== void 0 ? _ruleParams$anomalyDe : []))]
            }
          },
          aggs: {
            anomaly_groups: {
              multi_terms: {
                terms: [{
                  field: 'partition_field_value'
                }, {
                  field: 'by_field_value'
                }, {
                  field: 'job_id'
                }, {
                  field: 'detector_index'
                }],
                size: 1000,
                order: {
                  'latest_score.record_score': 'desc'
                }
              },
              aggs: {
                latest_score: {
                  top_metrics: {
                    metrics: (0, _as_mutable_array.asMutableArray)([{
                      field: 'record_score'
                    }, {
                      field: 'partition_field_value'
                    }, {
                      field: 'by_field_value'
                    }, {
                      field: 'job_id'
                    }, {
                      field: 'timestamp'
                    }, {
                      field: 'bucket_span'
                    }, {
                      field: 'detector_index'
                    }]),
                    sort: {
                      timestamp: 'desc'
                    }
                  }
                }
              }
            }
          }
        }
      };
      const response = await mlAnomalySearch(anomalySearchParams, []);
      const anomalies = (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.anomaly_groups.buckets.map(bucket => {
        const latest = bucket.latest_score.top[0].metrics;
        const job = mlJobs.find(j => j.jobId === latest.job_id);
        if (!job) {
          logger.warn(`Could not find matching job for job id ${latest.job_id}`);
          return undefined;
        }
        return {
          serviceName: latest.partition_field_value,
          transactionType: latest.by_field_value,
          environment: job.environment,
          score: latest.record_score,
          detectorType: (0, _apm_ml_detectors.getAnomalyDetectorType)(latest.detector_index),
          timestamp: Date.parse(latest.timestamp),
          bucketSpan: latest.bucket_span,
          bucketKey: bucket.key
        };
      }).filter(anomaly => anomaly ? anomaly.score >= threshold : false)) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
      await (0, _std.asyncForEach)((0, _lodash.compact)(anomalies), async anomaly => {
        var _getEnvironmentEsFiel;
        const {
          serviceName,
          environment,
          transactionType,
          score,
          detectorType,
          timestamp,
          bucketSpan,
          bucketKey
        } = anomaly;
        const eventSourceFields = await (0, _get_service_group_fields_for_anomaly.getServiceGroupFieldsForAnomaly)({
          apmIndices,
          scopedClusterClient,
          savedObjectsClient,
          serviceName,
          environment,
          transactionType,
          timestamp,
          bucketSpan
        });
        const severityLevel = (0, _anomaly_detection.getSeverity)(score);
        const reasonMessage = (0, _apm_rule_types.formatAnomalyReason)({
          anomalyScore: score,
          serviceName,
          severityLevel,
          windowSize: params.windowSize,
          windowUnit: params.windowUnit,
          detectorType
        });
        const alertId = bucketKey.join('_');
        const {
          uuid,
          start
        } = alertsClient.report({
          id: alertId,
          actionGroup: ruleTypeConfig.defaultActionGroupId
        });
        const indexedStartedAt = start !== null && start !== void 0 ? start : startedAt.toISOString();
        const relativeViewInAppUrl = (0, _formatters.getAlertUrlTransaction)(serviceName, (_getEnvironmentEsFiel = (0, _environment_filter_values.getEnvironmentEsField)(environment)) === null || _getEnvironmentEsFiel === void 0 ? void 0 : _getEnvironmentEsFiel[_apm.SERVICE_ENVIRONMENT], transactionType);
        const viewInAppUrl = (0, _common2.addSpaceIdToPath)(basePath.publicBaseUrl, spaceId, relativeViewInAppUrl);
        const alertDetailsUrl = await (0, _common.getAlertUrl)(uuid, spaceId, indexedStartedAt, alertsLocator, basePath.publicBaseUrl);
        const payload = {
          [_apm.SERVICE_NAME]: serviceName,
          ...(0, _environment_filter_values.getEnvironmentEsField)(environment),
          [_apm.TRANSACTION_TYPE]: transactionType,
          [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction,
          [_ruleDataUtils.ALERT_SEVERITY]: severityLevel,
          [_ruleDataUtils.ALERT_EVALUATION_VALUE]: score,
          [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: threshold,
          [_ruleDataUtils.ALERT_REASON]: reasonMessage,
          ...eventSourceFields
        };
        const context = {
          alertDetailsUrl,
          environment: (0, _environment_filter_values.getEnvironmentLabel)(environment),
          reason: reasonMessage,
          serviceName,
          threshold: selectedOption === null || selectedOption === void 0 ? void 0 : selectedOption.label,
          transactionType,
          triggerValue: severityLevel,
          viewInAppUrl
        };
        alertsClient.setAlertData({
          id: alertId,
          payload,
          context
        });
      });
      return {
        state: {}
      };
    },
    alerts: _register_apm_rule_types.ApmRuleTypeAlertDefinition,
    getViewInAppRelativeUrl: ({
      rule
    }) => _common.observabilityPaths.ruleDetails(rule.id)
  });
}