"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findRulesRoute = exports.findInternalRulesRoute = void 0;
var _lib = require("../../../lib");
var _find = require("../../../../../common/routes/rule/apis/find");
var _types = require("../../../../types");
var _track_legacy_terminology = require("../../../lib/track_legacy_terminology");
var _transforms = require("./transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildFindRulesRoute = ({
  licenseState,
  path,
  router,
  excludeFromPublicApi = false,
  usageCounter
}) => {
  router.get({
    path,
    options: {
      access: 'public',
      summary: 'Get information about rules',
      tags: ['oas-tag:alerting']
    },
    validate: {
      query: _find.findRulesRequestQuerySchemaV1
    }
  }, router.handleLegacyErrors((0, _lib.verifyAccessAndContext)(licenseState, async function (context, req, res) {
    const rulesClient = (await context.alerting).getRulesClient();
    const query = req.query;
    (0, _track_legacy_terminology.trackLegacyTerminology)([query.search, query.search_fields, query.sort_field].filter(Boolean), usageCounter);
    const options = (0, _transforms.transformFindRulesBodyV1)({
      ...query,
      has_reference: query.has_reference || undefined,
      search_fields: searchFieldsAsArray(query.search_fields)
    });
    if (req.query.fields) {
      usageCounter === null || usageCounter === void 0 ? void 0 : usageCounter.incrementCounter({
        counterName: `alertingFieldsUsage`,
        counterType: 'alertingFieldsUsage',
        incrementBy: 1
      });
    }
    const findResult = await rulesClient.find({
      options,
      excludeFromPublicApi,
      includeSnoozeData: true
    });
    const responseBody = (0, _transforms.transformFindRulesResponseV1)(findResult, options.fields);
    return res.ok({
      body: responseBody
    });
  })));
  if (path === _types.INTERNAL_ALERTING_API_FIND_RULES_PATH) {
    router.post({
      path,
      validate: {
        body: _find.findRulesRequestQuerySchemaV1
      }
    }, router.handleLegacyErrors((0, _lib.verifyAccessAndContext)(licenseState, async function (context, req, res) {
      const rulesClient = (await context.alerting).getRulesClient();
      const body = req.body;
      (0, _track_legacy_terminology.trackLegacyTerminology)([req.body.search, req.body.search_fields, req.body.sort_field].filter(Boolean), usageCounter);
      const options = (0, _transforms.transformFindRulesBodyV1)({
        ...body,
        has_reference: body.has_reference || undefined,
        search_fields: searchFieldsAsArray(body.search_fields)
      });
      if (req.body.fields) {
        usageCounter === null || usageCounter === void 0 ? void 0 : usageCounter.incrementCounter({
          counterName: `alertingFieldsUsage`,
          counterType: 'alertingFieldsUsage',
          incrementBy: 1
        });
      }
      const findResult = await rulesClient.find({
        options,
        excludeFromPublicApi,
        includeSnoozeData: true
      });
      const responseBody = (0, _transforms.transformFindRulesResponseV1)(findResult, options.fields);
      return res.ok({
        body: responseBody
      });
    })));
  }
};
const findRulesRoute = (router, licenseState, usageCounter) => {
  buildFindRulesRoute({
    excludeFromPublicApi: true,
    licenseState,
    path: `${_types.BASE_ALERTING_API_PATH}/rules/_find`,
    router,
    usageCounter
  });
};
exports.findRulesRoute = findRulesRoute;
const findInternalRulesRoute = (router, licenseState, usageCounter) => {
  buildFindRulesRoute({
    excludeFromPublicApi: false,
    licenseState,
    path: _types.INTERNAL_ALERTING_API_FIND_RULES_PATH,
    router,
    usageCounter
  });
};
exports.findInternalRulesRoute = findInternalRulesRoute;
function searchFieldsAsArray(searchFields) {
  if (!searchFields) {
    return;
  }
  return Array.isArray(searchFields) ? searchFields : [searchFields];
}