"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TASK_ID = void 0;
exports.getApiKeyIdsToInvalidate = getApiKeyIdsToInvalidate;
exports.getFindFilter = getFindFilter;
exports.initializeApiKeyInvalidator = initializeApiKeyInvalidator;
exports.invalidateApiKeysAndDeletePendingApiKeySavedObject = invalidateApiKeysAndDeletePendingApiKeySavedObject;
exports.runInvalidate = runInvalidate;
exports.scheduleApiKeyInvalidatorTask = scheduleApiKeyInvalidatorTask;
exports.taskRunner = taskRunner;
var _get_cadence = require("../lib/get_cadence");
var _task_state = require("./task_state");
var _ = require("..");
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'alerts_invalidate_api_keys';
const PAGE_SIZE = 100;
const TASK_ID = exports.TASK_ID = `Alerts-${TASK_TYPE}`;
const invalidateAPIKeys = async (params, securityPluginStart) => {
  if (!securityPluginStart) {
    return {
      apiKeysEnabled: false
    };
  }
  const invalidateAPIKeyResult = await securityPluginStart.authc.apiKeys.invalidateAsInternalUser(params);
  // Null when Elasticsearch security is disabled
  if (!invalidateAPIKeyResult) {
    return {
      apiKeysEnabled: false
    };
  }
  return {
    apiKeysEnabled: true,
    result: invalidateAPIKeyResult
  };
};
function initializeApiKeyInvalidator(logger, coreStartServices, taskManager, config) {
  registerApiKeyInvalidatorTaskDefinition(logger, coreStartServices, taskManager, config);
}
async function scheduleApiKeyInvalidatorTask(logger, config, taskManager) {
  const interval = config.invalidateApiKeysTask.interval;
  try {
    await taskManager.ensureScheduled({
      id: TASK_ID,
      taskType: TASK_TYPE,
      schedule: {
        interval
      },
      state: _task_state.emptyState,
      params: {}
    });
  } catch (e) {
    logger.error(`Error scheduling ${TASK_ID} task, received ${e.message}`);
  }
}
function registerApiKeyInvalidatorTaskDefinition(logger, coreStartServices, taskManager, config) {
  taskManager.registerTaskDefinitions({
    [TASK_TYPE]: {
      title: 'Invalidate alert API Keys',
      stateSchemaByVersion: _task_state.stateSchemaByVersion,
      createTaskRunner: taskRunner(logger, coreStartServices, config)
    }
  });
}
function taskRunner(logger, coreStartServices, config) {
  return ({
    taskInstance
  }) => {
    const state = taskInstance.state;
    return {
      async run() {
        let totalInvalidated = 0;
        try {
          const [{
            savedObjects
          }, {
            encryptedSavedObjects,
            security
          }] = await coreStartServices;
          const savedObjectsClient = savedObjects.createInternalRepository([_.API_KEY_PENDING_INVALIDATION_TYPE, _saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE]);
          const encryptedSavedObjectsClient = encryptedSavedObjects.getClient({
            includedHiddenTypes: [_.API_KEY_PENDING_INVALIDATION_TYPE]
          });
          totalInvalidated = await runInvalidate({
            config,
            encryptedSavedObjectsClient,
            logger,
            savedObjectsClient,
            security
          });
          const updatedState = {
            runs: (state.runs || 0) + 1,
            total_invalidated: totalInvalidated
          };
          return {
            state: updatedState,
            schedule: {
              interval: config.invalidateApiKeysTask.interval
            }
          };
        } catch (e) {
          logger.warn(`Error executing alerting apiKey invalidation task: ${e.message}`);
          const updatedState = {
            runs: state.runs + 1,
            total_invalidated: totalInvalidated
          };
          return {
            state: updatedState,
            schedule: {
              interval: config.invalidateApiKeysTask.interval
            }
          };
        }
      }
    };
  };
}
async function runInvalidate({
  config,
  encryptedSavedObjectsClient,
  logger,
  savedObjectsClient,
  security
}) {
  const configuredDelay = config.invalidateApiKeysTask.removalDelay;
  const delay = (0, _get_cadence.timePeriodBeforeDate)(new Date(), configuredDelay).toISOString();
  let hasMoreApiKeysPendingInvalidation = true;
  let totalInvalidated = 0;
  const excludedSOIds = new Set();
  do {
    // Query for PAGE_SIZE api keys to invalidate at a time. At the end of each iteration,
    // we should have deleted the deletable keys and added keys still in use to the excluded list
    const filter = getFindFilter(delay, [...excludedSOIds]);
    const apiKeysToInvalidate = await savedObjectsClient.find({
      type: _.API_KEY_PENDING_INVALIDATION_TYPE,
      filter,
      page: 1,
      sortField: 'createdAt',
      sortOrder: 'asc',
      perPage: PAGE_SIZE
    });
    if (apiKeysToInvalidate.total > 0) {
      const {
        apiKeyIdsToExclude,
        apiKeyIdsToInvalidate
      } = await getApiKeyIdsToInvalidate({
        apiKeySOsPendingInvalidation: apiKeysToInvalidate,
        encryptedSavedObjectsClient,
        savedObjectsClient
      });
      apiKeyIdsToExclude.forEach(({
        id
      }) => excludedSOIds.add(id));
      totalInvalidated += await invalidateApiKeysAndDeletePendingApiKeySavedObject({
        apiKeyIdsToInvalidate,
        logger,
        savedObjectsClient,
        securityPluginStart: security
      });
    }
    hasMoreApiKeysPendingInvalidation = apiKeysToInvalidate.total > PAGE_SIZE;
  } while (hasMoreApiKeysPendingInvalidation);
  return totalInvalidated;
}
async function getApiKeyIdsToInvalidate({
  apiKeySOsPendingInvalidation,
  encryptedSavedObjectsClient,
  savedObjectsClient
}) {
  var _ref, _aggregations$apiKeyI;
  // Decrypt the apiKeyId for each pending invalidation SO
  const apiKeyIds = await Promise.all(apiKeySOsPendingInvalidation.saved_objects.map(async apiKeyPendingInvalidationSO => {
    const decryptedApiKeyPendingInvalidationObject = await encryptedSavedObjectsClient.getDecryptedAsInternalUser(_.API_KEY_PENDING_INVALIDATION_TYPE, apiKeyPendingInvalidationSO.id);
    return {
      id: decryptedApiKeyPendingInvalidationObject.id,
      apiKeyId: decryptedApiKeyPendingInvalidationObject.attributes.apiKeyId
    };
  }));

  // Query saved objects index to see if any API keys are in use
  const filter = `${apiKeyIds.map(({
    apiKeyId
  }) => `${_saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE}.attributes.apiKeyId: "${apiKeyId}"`).join(' OR ')}`;
  const {
    aggregations
  } = await savedObjectsClient.find({
    type: _saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE,
    filter,
    perPage: 0,
    namespaces: ['*'],
    aggs: {
      apiKeyId: {
        terms: {
          field: `${_saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE}.attributes.apiKeyId`,
          size: PAGE_SIZE
        }
      }
    }
  });
  const apiKeyIdsInUseBuckets = (_ref = aggregations === null || aggregations === void 0 ? void 0 : (_aggregations$apiKeyI = aggregations.apiKeyId) === null || _aggregations$apiKeyI === void 0 ? void 0 : _aggregations$apiKeyI.buckets) !== null && _ref !== void 0 ? _ref : [];
  const apiKeyIdsToInvalidate = [];
  const apiKeyIdsToExclude = [];
  apiKeyIds.forEach(({
    id,
    apiKeyId
  }) => {
    if (apiKeyIdsInUseBuckets.find(bucket => bucket.key === apiKeyId)) {
      apiKeyIdsToExclude.push({
        id,
        apiKeyId
      });
    } else {
      apiKeyIdsToInvalidate.push({
        id,
        apiKeyId
      });
    }
  });
  return {
    apiKeyIdsToInvalidate,
    apiKeyIdsToExclude
  };
}
function getFindFilter(delay, excludedSOIds = []) {
  let filter = `${_.API_KEY_PENDING_INVALIDATION_TYPE}.attributes.createdAt <= "${delay}"`;
  if (excludedSOIds.length > 0) {
    const excluded = [...new Set(excludedSOIds)];
    const excludedSOIdFilter = (excluded !== null && excluded !== void 0 ? excluded : []).map(id => `NOT ${_.API_KEY_PENDING_INVALIDATION_TYPE}.id: "${_.API_KEY_PENDING_INVALIDATION_TYPE}:${id}"`);
    filter += ` AND ${excludedSOIdFilter.join(' AND ')}`;
  }
  return filter;
}
async function invalidateApiKeysAndDeletePendingApiKeySavedObject({
  apiKeyIdsToInvalidate,
  logger,
  savedObjectsClient,
  securityPluginStart
}) {
  let totalInvalidated = 0;
  if (apiKeyIdsToInvalidate.length > 0) {
    const ids = apiKeyIdsToInvalidate.map(({
      apiKeyId
    }) => apiKeyId);
    const response = await invalidateAPIKeys({
      ids
    }, securityPluginStart);
    if (response.apiKeysEnabled === true && response.result.error_count > 0) {
      logger.error(`Failed to invalidate API Keys [ids="${ids.join(', ')}"]`);
    } else {
      await Promise.all(apiKeyIdsToInvalidate.map(async ({
        id,
        apiKeyId
      }) => {
        try {
          await savedObjectsClient.delete(_.API_KEY_PENDING_INVALIDATION_TYPE, id);
          totalInvalidated++;
        } catch (err) {
          logger.error(`Failed to delete invalidated API key "${apiKeyId}". Error: ${err.message}`);
        }
      }));
    }
  }
  logger.debug(`Total invalidated API keys "${totalInvalidated}"`);
  return totalInvalidated;
}