"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.REFRESH_FIELDS_ALL = void 0;
exports.resolveAlertConflicts = resolveAlertConflicts;
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _lodash = require("lodash");
var _2 = require("../..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// these fields are the one's we'll refresh from the fresh mget'd docs
const REFRESH_FIELDS_ALWAYS = [_ruleDataUtils.ALERT_WORKFLOW_STATUS, _ruleDataUtils.ALERT_WORKFLOW_TAGS, _ruleDataUtils.ALERT_CASE_IDS];
const REFRESH_FIELDS_CONDITIONAL = [_ruleDataUtils.ALERT_STATUS];
const REFRESH_FIELDS_ALL = exports.REFRESH_FIELDS_ALL = [...REFRESH_FIELDS_ALWAYS, ...REFRESH_FIELDS_CONDITIONAL];
// wrapper to catch anything thrown; current usage of this function is
// to replace just logging that the error occurred, so we don't want
// to cause _more_ errors ...
async function resolveAlertConflicts(params) {
  const {
    logger
  } = params;
  try {
    await resolveAlertConflicts_(params);
  } catch (err) {
    logger.error(`Error resolving alert conflicts: ${err.message}`);
  }
}
async function resolveAlertConflicts_(params) {
  var _bulkRequest$operatio, _bulkResponse$items, _mbrResponse$bulkResp;
  const {
    logger,
    esClient,
    bulkRequest,
    bulkResponse
  } = params;
  if (bulkRequest.operations && ((_bulkRequest$operatio = bulkRequest.operations) === null || _bulkRequest$operatio === void 0 ? void 0 : _bulkRequest$operatio.length) === 0) return;
  if (bulkResponse.items && ((_bulkResponse$items = bulkResponse.items) === null || _bulkResponse$items === void 0 ? void 0 : _bulkResponse$items.length) === 0) return;

  // get numbers for a summary log message
  const {
    success,
    errors,
    conflicts,
    messages
  } = getResponseStats(bulkResponse);
  if (conflicts === 0 && errors === 0) return;
  const allMessages = messages.join('; ');
  logger.error(`Error writing alerts: ${success} successful, ${conflicts} conflicts, ${errors} errors: ${allMessages}`);

  // get a new bulk request for just conflicted docs
  const conflictRequest = getConflictRequest(bulkRequest, bulkResponse);
  if (conflictRequest.length === 0) return;

  // get the fresh versions of those docs
  const freshDocs = await getFreshDocs(esClient, conflictRequest);

  // update the OCC and refresh-able fields
  await updateOCC(conflictRequest, freshDocs);
  await refreshFieldsInDocs(conflictRequest, freshDocs);
  logger.info(`Retrying bulk update of ${conflictRequest.length} conflicted alerts`);
  const mbrResponse = await makeBulkRequest(params.esClient, params.bulkRequest, conflictRequest);
  if (((_mbrResponse$bulkResp = mbrResponse.bulkResponse) === null || _mbrResponse$bulkResp === void 0 ? void 0 : _mbrResponse$bulkResp.items.length) !== conflictRequest.length) {
    var _mbrResponse$bulkResp2;
    const actual = (_mbrResponse$bulkResp2 = mbrResponse.bulkResponse) === null || _mbrResponse$bulkResp2 === void 0 ? void 0 : _mbrResponse$bulkResp2.items.length;
    const expected = conflictRequest.length;
    logger.error(`Unexpected number of bulk response items retried; expecting ${expected}, retried ${actual}`);
    return;
  }
  if (mbrResponse.error) {
    const index = bulkRequest.index || 'unknown index';
    logger.error(`Error writing ${conflictRequest.length} alerts to ${index} - ${mbrResponse.error.message}`);
    return;
  }
  if (mbrResponse.errors === 0) {
    logger.info(`Retried bulk update of ${conflictRequest.length} conflicted alerts succeeded`);
  } else {
    logger.error(`Retried bulk update of ${conflictRequest.length} conflicted alerts still had ${mbrResponse.errors} conflicts`);
  }
}
// make the bulk request to fix conflicts
async function makeBulkRequest(esClient, bulkRequest, conflictRequest) {
  const operations = conflictRequest.map(req => [req.op, req.doc]).flat();
  // just replace the operations from the original request
  const updatedBulkRequest = {
    ...bulkRequest,
    operations
  };
  const bulkResponse = await esClient.bulk(updatedBulkRequest);
  const errors = bulkResponse.items.filter(item => {
    var _item$index;
    return (_item$index = item.index) === null || _item$index === void 0 ? void 0 : _item$index.error;
  }).length;
  return {
    bulkRequest,
    bulkResponse,
    errors
  };
}

/** Update refreshable fields in the conflict requests. */
async function refreshFieldsInDocs(conflictRequests, freshResponses) {
  for (const [conflictRequest, freshResponse] of (0, _lodash.zip)(conflictRequests, freshResponses)) {
    if (!(conflictRequest !== null && conflictRequest !== void 0 && conflictRequest.op.index) || !freshResponse) continue;

    // @ts-expect-error @elastic/elasticsearch _source is not in the type!
    const freshDoc = freshResponse._source;
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const conflictDoc = conflictRequest.doc;
    if (!freshDoc || !conflictDoc) continue;
    for (const refreshField of REFRESH_FIELDS_ALWAYS) {
      const val = (0, _lodash.get)(freshDoc, refreshField);
      conflictDoc[refreshField] = val;
    }

    // structured this way to make sure all conditional refresh
    // fields are listed in REFRESH_FIELDS_CONDITIONAL when we mget
    for (const refreshField of REFRESH_FIELDS_CONDITIONAL) {
      switch (refreshField) {
        // hamdling for kibana.alert.status: overwrite conflict doc
        // with fresh version if it's not active or recovered (ie, untracked)
        case _ruleDataUtils.ALERT_STATUS:
          const freshStatus = (0, _lodash.get)(freshDoc, _ruleDataUtils.ALERT_STATUS);
          if (freshStatus !== _ruleDataUtils.ALERT_STATUS_ACTIVE && freshStatus !== _ruleDataUtils.ALERT_STATUS_RECOVERED) {
            conflictDoc[_ruleDataUtils.ALERT_STATUS] = freshStatus;
          }
          break;
      }
    }
  }
}

/** Update the OCC info in the conflict request with the fresh info. */
async function updateOCC(conflictRequests, freshDocs) {
  for (const [req, freshDoc] of (0, _lodash.zip)(conflictRequests, freshDocs)) {
    if (!(req !== null && req !== void 0 && req.op.index) || !freshDoc) continue;

    // @ts-expect-error @elastic/elasticsearch _seq_no is not in the type!
    const seqNo = freshDoc._seq_no;
    // @ts-expect-error @elastic/elasticsearch _primary_term is not in the type!
    const primaryTerm = freshDoc._primary_term;
    if (seqNo === undefined) throw new Error('Unexpected undefined seqNo');
    if (primaryTerm === undefined) throw new Error('Unexpected undefined primaryTerm');
    req.op.index.if_seq_no = seqNo;
    req.op.index.if_primary_term = primaryTerm;
  }
}

/** Get the latest version of the conflicted docs, with fields to refresh. */
async function getFreshDocs(esClient, conflictRequests) {
  const docs = [];
  conflictRequests.forEach(req => {
    var _req$op$index, _req$op$index2;
    const [id, index] = [(_req$op$index = req.op.index) === null || _req$op$index === void 0 ? void 0 : _req$op$index._id, (_req$op$index2 = req.op.index) === null || _req$op$index2 === void 0 ? void 0 : _req$op$index2._index];
    if (!id || !index) return;
    docs.push({
      _id: id,
      _index: index
    });
  });
  const mgetRes = await esClient.mget({
    docs,
    _source_includes: REFRESH_FIELDS_ALL
  });
  if (mgetRes.docs.length !== docs.length) {
    throw new Error(`Unexpected number of mget response docs; expected ${docs.length}, got ${mgetRes.docs.length}`);
  }
  return mgetRes.docs;
}

/** Return the bulk request, filtered to those requests that had conflicts. */
function getConflictRequest(bulkRequest, bulkResponse) {
  // first "normalize" the request from it's non-linear form
  const request = normalizeRequest(bulkRequest);

  // maybe we didn't unwind it right ...
  if (request.length !== bulkResponse.items.length) {
    throw new Error('Unexpected number of bulk response items');
  }
  if (request.length === 0) return [];

  // we only want op: index where the status was 409 / conflict
  const conflictRequest = (0, _lodash.zip)(request, bulkResponse.items).filter(([_, res]) => {
    var _res$index;
    return (res === null || res === void 0 ? void 0 : (_res$index = res.index) === null || _res$index === void 0 ? void 0 : _res$index.status) === 409;
  }).map(([req, _]) => req);
  return conflictRequest;
}

/** Convert a bulk request (op | doc)[] to an array of { op, doc }[]  */
function normalizeRequest(bulkRequest) {
  if (!bulkRequest.operations) return [];
  const result = [];
  let index = 0;
  while (index < bulkRequest.operations.length) {
    // the "op" data
    const op = bulkRequest.operations[index];

    // now the "doc" data, if there is any (none for delete)
    if (op.create || op.index || op.update) {
      index++;
      const doc = bulkRequest.operations[index];
      result.push({
        op,
        doc
      });
    } else if (op.delete) {
      // no doc for delete op
    } else {
      throw new Error(`Unsupported bulk operation: ${JSON.stringify(op)}`);
    }
    index++;
  }
  return result;
}
// generate a summary of the original bulk request attempt, for logging
function getResponseStats(bulkResponse) {
  const sanitizedResponse = (0, _2.sanitizeBulkErrorResponse)(bulkResponse);
  const stats = {
    success: 0,
    conflicts: 0,
    errors: 0,
    messages: []
  };
  for (const item of sanitizedResponse.items) {
    const op = item.create || item.index || item.update || item.delete;
    if (op !== null && op !== void 0 && op.error) {
      if ((op === null || op === void 0 ? void 0 : op.status) === 409 && op === item.index) {
        stats.conflicts++;
      } else {
        var _op$error;
        stats.errors++;
        stats.messages.push((op === null || op === void 0 ? void 0 : (_op$error = op.error) === null || _op$error === void 0 ? void 0 : _op$error.reason) || 'no bulk reason provided');
      }
    } else {
      stats.success++;
    }
  }
  return stats;
}