"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchFrequentItemSets = fetchFrequentItemSets;
exports.getFrequentItemSetsAggFields = getFrequentItemSetsAggFields;
exports.getShouldClauses = getShouldClauses;
exports.groupDuplicates = groupDuplicates;
var _lodash = require("lodash");
var _d3Array = require("d3-array");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function groupDuplicates(cps, uniqueFields) {
  const groups = [];
  for (const cp of cps) {
    const compareAttributes = (0, _lodash.pick)(cp, uniqueFields);
    const groupIndex = groups.findIndex(g => (0, _lodash.isEqual)(g.keys, compareAttributes));
    if (groupIndex === -1) {
      groups.push({
        keys: compareAttributes,
        group: [cp]
      });
    } else {
      groups[groupIndex].group.push(cp);
    }
  }
  return groups;
}

/**
 * Creates ES bool should clauses for each provided significant item.
 * In previous versions of this helper we grouped values for the same field
 * in a `terms` agg, but this might clash with the `minimum_should_match: 2` clause
 * used in the query for the `frequent_item_sets`, because even multiple matches within
 * the `terms` agg would count as just 1 match in the outer `should` part.
 *
 * @param significantItems
 * @returns an array of term filters
 */
function getShouldClauses(significantItems) {
  return significantItems.map(d => ({
    term: {
      [d.fieldName]: d.fieldValue
    }
  }));
}

/**
 * Creates a filter for each field to be used in the `frequent_items_sets` agg.
 * Considers a limit per field to work around scaling limitations of the agg.
 *
 * @param significantItems
 * @returns field filter for the `frequent_item_sets` agg
 */
function getFrequentItemSetsAggFields(significantItems) {
  return Array.from((0, _d3Array.group)(significantItems, ({
    fieldName
  }) => fieldName), ([field, values]) => ({
    field,
    include: values.map(d => String(d.fieldValue)).slice(0, _constants.LOG_RATE_ANALYSIS_SETTINGS.FREQUENT_ITEMS_SETS_FIELD_VALUE_LIMIT)
  }));
}
async function fetchFrequentItemSets(client, index, searchQuery, significantItems, timeFieldName, deviationMin, deviationMax, logger,
// The default value of 1 means no sampling will be used
sampleProbability = 1, emitError, abortSignal) {
  // Sort significant terms by ascending p-value, necessary to apply the field limit correctly.
  const sortedSignificantItems = significantItems.slice().sort((a, b) => {
    var _a$pValue, _b$pValue;
    return ((_a$pValue = a.pValue) !== null && _a$pValue !== void 0 ? _a$pValue : 0) - ((_b$pValue = b.pValue) !== null && _b$pValue !== void 0 ? _b$pValue : 0);
  });
  const query = {
    bool: {
      minimum_should_match: 2,
      filter: [searchQuery, {
        range: {
          [timeFieldName]: {
            gte: deviationMin,
            lt: deviationMax
          }
        }
      }],
      should: getShouldClauses(sortedSignificantItems)
    }
  };
  const frequentItemSetsAgg = {
    fi: {
      frequent_item_sets: {
        minimum_set_size: 2,
        size: 200,
        minimum_support: _constants.LOG_RATE_ANALYSIS_SETTINGS.FREQUENT_ITEMS_SETS_MINIMUM_SUPPORT,
        fields: getFrequentItemSetsAggFields(sortedSignificantItems)
      }
    }
  };
  const {
    wrap,
    unwrap
  } = (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
    probability: sampleProbability,
    seed: _constants.RANDOM_SAMPLER_SEED
  });
  const esBody = {
    query,
    aggs: wrap(frequentItemSetsAgg),
    size: 0,
    track_total_hits: true
  };
  const body = await client.search({
    index,
    size: 0,
    body: esBody
  }, {
    signal: abortSignal,
    maxRetries: 0
  });
  if (body.aggregations === undefined) {
    logger.error(`Failed to fetch frequent_item_sets, got: \n${JSON.stringify(body, null, 2)}`);
    emitError(`Failed to fetch frequent_item_sets.`);
    return {
      fields: [],
      itemSets: [],
      totalDocCount: 0
    };
  }
  const totalDocCountFi = body.hits.total.value;
  const frequentItemSets = unwrap(body.aggregations);
  const shape = frequentItemSets.fi.buckets.length;
  let maximum = shape;
  if (maximum > 50000) {
    maximum = 50000;
  }
  const fiss = frequentItemSets.fi.buckets;
  fiss.length = maximum;
  const results = [];
  fiss.forEach(fis => {
    const result = {
      set: [],
      size: 0,
      maxPValue: 0,
      doc_count: 0,
      support: 0,
      total_doc_count: 0
    };
    let maxPValue;
    Object.entries(fis.key).forEach(([key, values]) => {
      values.forEach(value => {
        var _sortedSignificantIte;
        result.set.push({
          fieldName: key,
          fieldValue: value
        });
        const pValue = (_sortedSignificantIte = sortedSignificantItems.find(t => t.fieldName === key && t.fieldValue === value)) === null || _sortedSignificantIte === void 0 ? void 0 : _sortedSignificantIte.pValue;
        if (pValue !== undefined && pValue !== null) {
          var _maxPValue;
          maxPValue = Math.max((_maxPValue = maxPValue) !== null && _maxPValue !== void 0 ? _maxPValue : 0, pValue);
        }
      });
    });
    if (maxPValue === undefined) {
      return;
    }
    result.size = Object.keys(result.set).length;
    result.maxPValue = maxPValue;
    result.doc_count = fis.doc_count;
    result.support = fis.support;
    result.total_doc_count = totalDocCountFi;
    results.push(result);
  });
  results.sort((a, b) => {
    return b.doc_count - a.doc_count;
  });
  const uniqueFields = (0, _lodash.uniq)(results.flatMap(r => r.set.map(d => d.fieldName)));
  return {
    fields: uniqueFields,
    itemSets: results,
    totalDocCount: totalDocCountFi
  };
}