"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAll = getAll;
exports.getAllSystemConnectors = getAllSystemConnectors;
exports.getAllUnsecured = getAllUnsecured;
var _lodash = require("lodash");
var _schemas = require("../../schemas");
var _connector = require("../../../../data/connector");
var _audit_events = require("../../../../lib/audit_events");
var _lib = require("../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get all actions with in-memory connectors
 */

async function getAll({
  context,
  includeSystemActions = false
}) {
  try {
    await context.authorization.ensureAuthorized({
      operation: 'get'
    });
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.connectorAuditEvent)({
      action: _audit_events.ConnectorAuditAction.FIND,
      error
    }));
    throw error;
  }
  return await getAllHelper({
    auditLogger: context.auditLogger,
    esClient: context.scopedClusterClient.asInternalUser,
    inMemoryConnectors: includeSystemActions ? context.inMemoryConnectors : context.inMemoryConnectors.filter(connector => !connector.isSystemAction),
    kibanaIndices: context.kibanaIndices,
    logger: context.logger,
    savedObjectsClient: context.unsecuredSavedObjectsClient
  });
}
async function getAllUnsecured({
  esClient,
  inMemoryConnectors,
  internalSavedObjectsRepository,
  kibanaIndices,
  logger,
  spaceId
}) {
  const namespace = spaceId && spaceId !== 'default' ? spaceId : undefined;
  return await getAllHelper({
    esClient,
    // Unsecured execution does not currently support system actions so we filter them out
    inMemoryConnectors: inMemoryConnectors.filter(connector => !connector.isSystemAction),
    kibanaIndices,
    logger,
    namespace,
    savedObjectsClient: internalSavedObjectsRepository
  });
}
async function getAllHelper({
  auditLogger,
  esClient,
  inMemoryConnectors,
  kibanaIndices,
  logger,
  namespace,
  savedObjectsClient
}) {
  const savedObjectsActions = (await (0, _connector.findConnectorsSo)({
    savedObjectsClient,
    namespace
  })).saved_objects.map(rawAction => {
    const connector = (0, _lib.connectorFromSavedObject)(rawAction, (0, _lib.isConnectorDeprecated)(rawAction.attributes));
    return (0, _lodash.omit)(connector, 'secrets');
  });
  if (auditLogger) {
    savedObjectsActions.forEach(({
      id
    }) => auditLogger.log((0, _audit_events.connectorAuditEvent)({
      action: _audit_events.ConnectorAuditAction.FIND,
      savedObject: {
        type: 'action',
        id
      }
    })));
  }
  const mergedResult = [...savedObjectsActions, ...inMemoryConnectors.map(inMemoryConnector => ({
    id: inMemoryConnector.id,
    actionTypeId: inMemoryConnector.actionTypeId,
    name: inMemoryConnector.name,
    isPreconfigured: inMemoryConnector.isPreconfigured,
    isDeprecated: (0, _lib.isConnectorDeprecated)(inMemoryConnector),
    isSystemAction: inMemoryConnector.isSystemAction
  }))].sort((a, b) => a.name.localeCompare(b.name));
  const connectors = await injectExtraFindData({
    kibanaIndices,
    esClient,
    connectors: mergedResult
  });
  validateConnectors(connectors, logger);
  return connectors;
}
const validateConnectors = (connectors, logger) => {
  connectors.forEach(connector => {
    // Try to validate the connectors, but don't throw.
    try {
      _schemas.connectorWithExtraFindDataSchema.validate(connector);
    } catch (e) {
      logger.warn(`Error validating connector: ${connector.id}, ${e}`);
    }
  });
};
async function getAllSystemConnectors({
  context
}) {
  try {
    await context.authorization.ensureAuthorized({
      operation: 'get'
    });
  } catch (error) {
    var _context$auditLogger2;
    (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.connectorAuditEvent)({
      action: _audit_events.ConnectorAuditAction.FIND,
      error
    }));
    throw error;
  }
  const systemConnectors = context.inMemoryConnectors.filter(connector => connector.isSystemAction);
  const transformedSystemConnectors = systemConnectors.map(systemConnector => ({
    id: systemConnector.id,
    actionTypeId: systemConnector.actionTypeId,
    name: systemConnector.name,
    isPreconfigured: systemConnector.isPreconfigured,
    isDeprecated: (0, _lib.isConnectorDeprecated)(systemConnector),
    isSystemAction: systemConnector.isSystemAction
  })).sort((a, b) => a.name.localeCompare(b.name));
  const connectors = await injectExtraFindData({
    kibanaIndices: context.kibanaIndices,
    esClient: context.scopedClusterClient.asInternalUser,
    connectors: transformedSystemConnectors
  });
  validateConnectors(connectors, context.logger);
  return connectors;
}
async function injectExtraFindData({
  kibanaIndices,
  esClient,
  connectors
}) {
  const aggs = {};
  for (const connector of connectors) {
    aggs[connector.id] = {
      filter: {
        bool: {
          must: {
            nested: {
              path: 'references',
              query: {
                bool: {
                  filter: {
                    bool: {
                      must: [{
                        term: {
                          'references.id': connector.id
                        }
                      }, {
                        term: {
                          'references.type': 'action'
                        }
                      }]
                    }
                  }
                }
              }
            }
          }
        }
      }
    };
  }
  const aggregationResult = await (0, _connector.searchConnectorsSo)({
    esClient,
    kibanaIndices,
    aggs
  });
  return connectors.map(connector => ({
    ...connector,
    // @ts-expect-error aggegation type is not specified
    referencedByCount: aggregationResult.aggregations[connector.id].doc_count
  }));
}