"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ActionTypeRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _i18n = require("@kbn/i18n");
var _common = require("../common");
var _lib = require("./lib");
var _bidirectional_connectors = require("./lib/bidirectional_connectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ActionTypeRegistry {
  constructor(constructorParams) {
    (0, _defineProperty2.default)(this, "taskManager", void 0);
    (0, _defineProperty2.default)(this, "actionTypes", new Map());
    (0, _defineProperty2.default)(this, "taskRunnerFactory", void 0);
    (0, _defineProperty2.default)(this, "actionsConfigUtils", void 0);
    (0, _defineProperty2.default)(this, "licenseState", void 0);
    (0, _defineProperty2.default)(this, "inMemoryConnectors", void 0);
    (0, _defineProperty2.default)(this, "licensing", void 0);
    /**
     * Returns true if the action type is a system action type
     */
    (0, _defineProperty2.default)(this, "isSystemActionType", actionTypeId => {
      var _this$actionTypes$get;
      return Boolean((_this$actionTypes$get = this.actionTypes.get(actionTypeId)) === null || _this$actionTypes$get === void 0 ? void 0 : _this$actionTypes$get.isSystemActionType);
    });
    this.taskManager = constructorParams.taskManager;
    this.taskRunnerFactory = constructorParams.taskRunnerFactory;
    this.actionsConfigUtils = constructorParams.actionsConfigUtils;
    this.licenseState = constructorParams.licenseState;
    this.inMemoryConnectors = constructorParams.inMemoryConnectors;
    this.licensing = constructorParams.licensing;
  }

  /**
   * Returns if the action type registry has the given action type registered
   */
  has(id) {
    return this.actionTypes.has(id);
  }

  /**
   * Throws error if action type is not enabled.
   */
  ensureActionTypeEnabled(id) {
    this.actionsConfigUtils.ensureActionTypeEnabled(id);
    // Important to happen last because the function will notify of feature usage at the
    // same time and it shouldn't notify when the action type isn't enabled
    this.licenseState.ensureLicenseForActionType(this.get(id));
  }

  /**
   * Returns true if action type is enabled in the config and a valid license is used.
   */
  isActionTypeEnabled(id, options = {
    notifyUsage: false
  }) {
    return this.actionsConfigUtils.isActionTypeEnabled(id) && this.licenseState.isLicenseValidForActionType(this.get(id), options).isValid === true;
  }

  /**
   * Returns true if action type is enabled or preconfigured.
   * An action type can be disabled but used with a preconfigured action.
   * This does not apply to system actions as those can be disabled.
   */
  isActionExecutable(actionId, actionTypeId, options = {
    notifyUsage: false
  }) {
    const validLicense = this.licenseState.isLicenseValidForActionType(this.get(actionTypeId), options).isValid;
    if (validLicense === false) return false;
    const actionTypeEnabled = this.isActionTypeEnabled(actionTypeId, options);
    const inMemoryConnector = this.inMemoryConnectors.find(connector => connector.id === actionId);
    return actionTypeEnabled || !actionTypeEnabled && ((inMemoryConnector === null || inMemoryConnector === void 0 ? void 0 : inMemoryConnector.isPreconfigured) === true || (inMemoryConnector === null || inMemoryConnector === void 0 ? void 0 : inMemoryConnector.isSystemAction) === true);
  }
  /**
   * Returns the kibana privileges of a system action type
   */
  getSystemActionKibanaPrivileges(actionTypeId, params) {
    var _actionType$getKibana, _actionType$getKibana2;
    const actionType = this.actionTypes.get(actionTypeId);
    if (!(actionType !== null && actionType !== void 0 && actionType.isSystemActionType)) {
      return [];
    }
    return (_actionType$getKibana = actionType === null || actionType === void 0 ? void 0 : (_actionType$getKibana2 = actionType.getKibanaPrivileges) === null || _actionType$getKibana2 === void 0 ? void 0 : _actionType$getKibana2.call(actionType, {
      params
    })) !== null && _actionType$getKibana !== void 0 ? _actionType$getKibana : [];
  }

  /**
   * Registers an action type to the action type registry
   */
  register(actionType) {
    if (this.has(actionType.id)) {
      throw new Error(_i18n.i18n.translate('xpack.actions.actionTypeRegistry.register.duplicateActionTypeErrorMessage', {
        defaultMessage: 'Action type "{id}" is already registered.',
        values: {
          id: actionType.id
        }
      }));
    }
    if (!actionType.supportedFeatureIds || actionType.supportedFeatureIds.length === 0) {
      throw new Error(_i18n.i18n.translate('xpack.actions.actionTypeRegistry.register.missingSupportedFeatureIds', {
        defaultMessage: 'At least one "supportedFeatureId" value must be supplied for connector type "{connectorTypeId}".',
        values: {
          connectorTypeId: actionType.id
        }
      }));
    }
    if (!(0, _common.areValidFeatures)(actionType.supportedFeatureIds)) {
      throw new Error(_i18n.i18n.translate('xpack.actions.actionTypeRegistry.register.invalidConnectorFeatureIds', {
        defaultMessage: 'Invalid feature ids "{ids}" for connector type "{connectorTypeId}".',
        values: {
          connectorTypeId: actionType.id,
          ids: actionType.supportedFeatureIds.join(',')
        }
      }));
    }
    if (!actionType.isSystemActionType && actionType.getKibanaPrivileges) {
      throw new Error(_i18n.i18n.translate('xpack.actions.actionTypeRegistry.register.invalidKibanaPrivileges', {
        defaultMessage: 'Kibana privilege authorization is only supported for system action types'
      }));
    }
    const maxAttempts = this.actionsConfigUtils.getMaxAttempts({
      actionTypeId: actionType.id,
      actionTypeMaxAttempts: actionType.maxAttempts
    });
    this.actionTypes.set(actionType.id, {
      ...actionType
    });
    this.taskManager.registerTaskDefinitions({
      [`actions:${actionType.id}`]: {
        title: actionType.name,
        maxAttempts,
        createTaskRunner: context => this.taskRunnerFactory.create(context)
      }
    });
    // No need to notify usage on basic action types
    if (actionType.minimumLicenseRequired !== 'basic') {
      this.licensing.featureUsage.register((0, _lib.getActionTypeFeatureUsageName)(actionType), actionType.minimumLicenseRequired);
    }
  }

  /**
   * Returns an action type, throws if not registered
   */
  get(id) {
    if (!this.has(id)) {
      throw _boom.default.badRequest(_i18n.i18n.translate('xpack.actions.actionTypeRegistry.get.missingActionTypeErrorMessage', {
        defaultMessage: 'Action type "{id}" is not registered.',
        values: {
          id
        }
      }));
    }
    return this.actionTypes.get(id);
  }

  /**
   * Returns a list of registered action types [{ id, name, enabled }], filtered by featureId if provided.
   */
  list(featureId) {
    return Array.from(this.actionTypes).filter(([_, actionType]) => featureId ? actionType.supportedFeatureIds.includes(featureId) : true)
    // Temporarily don't return SentinelOne and Crowdstrike connector for Security Solution Rule Actions
    .filter(([actionTypeId]) => featureId ? !(0, _bidirectional_connectors.isBidirectionalConnectorType)(actionTypeId) : true).map(([actionTypeId, actionType]) => ({
      id: actionTypeId,
      name: actionType.name,
      minimumLicenseRequired: actionType.minimumLicenseRequired,
      enabled: this.isActionTypeEnabled(actionTypeId),
      enabledInConfig: this.actionsConfigUtils.isActionTypeEnabled(actionTypeId),
      enabledInLicense: !!this.licenseState.isLicenseValidForActionType(actionType).isValid,
      supportedFeatureIds: actionType.supportedFeatureIds,
      isSystemActionType: !!actionType.isSystemActionType
    }));
  }

  /**
   * Returns the actions configuration utilities
   */
  getUtils() {
    return this.actionsConfigUtils;
  }
  getAllTypes() {
    return [...this.list().map(({
      id
    }) => id)];
  }
}
exports.ActionTypeRegistry = ActionTypeRegistry;