import { getEnvironmentVariable } from "./utils/env.js";
function _getApiKey(apiKey) {
    return (apiKey ??
        getEnvironmentVariable("LANGCHAIN_HUB_API_KEY") ??
        getEnvironmentVariable("LANGCHAIN_API_KEY"));
}
function _getApiUrl(apiUrl) {
    const defaultApiUrl = "https://api.hub.langchain.com";
    const _apiUrl = apiUrl ?? getEnvironmentVariable("LANGCHAIN_HUB_API_URL") ?? defaultApiUrl;
    if (!_apiUrl) {
        throw new Error("LangChain Hub API URL cannot be empty");
    }
    return _apiUrl;
}
function parseOwnerRepoCommit(identifier) {
    let ownerRepo = identifier;
    let commit;
    if (identifier.includes(":")) {
        [ownerRepo, commit] = identifier.split(":", 2);
    }
    if (!ownerRepo.includes("/")) {
        throw new Error(`Invalid identifier ${identifier}. Identifier must be in the format of 'owner/repo:commit' or 'owner/repo'.`);
    }
    const [owner, repo] = ownerRepo.split("/");
    return [owner, repo, commit];
}
export class Client {
    constructor(config) {
        Object.defineProperty(this, "apiKey", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "apiUrl", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.apiKey = _getApiKey(config?.apiKey);
        this.apiUrl = _getApiUrl(config?.apiUrl);
    }
    get _hostUrl() {
        if (this.apiUrl.includes("localhost") ||
            this.apiUrl.includes("127.0.0.1") ||
            this.apiUrl.includes("::1")) {
            return "http://localhost";
        }
        else if (this.apiUrl.split(".")[0].includes("beta")) {
            return "https://beta.smith.langchain.com";
        }
        else if (this.apiUrl.split(".")[0].includes("dev")) {
            return "https://dev.smith.langchain.com";
        }
        else {
            return "https://smith.langchain.com";
        }
    }
    _getHeaders(method) {
        const headers = {};
        if (this.apiKey !== undefined) {
            headers["x-api-key"] = this.apiKey;
        }
        if (method === "POST" || method === "PATCH") {
            headers["content-type"] = "application/json";
        }
        return headers;
    }
    async getSettings() {
        const res = await fetch(`${this.apiUrl}/settings`, {
            headers: this._getHeaders(),
        });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        return json;
    }
    async setTenantHandle(tenantHandle) {
        const res = await fetch(`${this.apiUrl}/settings/handle`, {
            method: "POST",
            headers: this._getHeaders("POST"),
            body: JSON.stringify({ tenant_handle: tenantHandle }),
        });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        return json;
    }
    async listRepos(options) {
        const { limit = 100, offset = 0 } = options ?? {};
        const res = await fetch(`${this.apiUrl}/repos?limit=${limit}&offset=${offset}`, { headers: this._getHeaders() });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        return json;
    }
    async getRepo(repoFullName) {
        const res = await fetch(`${this.apiUrl}/repos/${repoFullName}`, {
            headers: this._getHeaders(),
        });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        return json;
    }
    async createRepo(repoHandle, options) {
        const { isPublic = true, description = "" } = options ?? {};
        const res = await fetch(`${this.apiUrl}/repos/`, {
            method: "POST",
            headers: this._getHeaders("POST"),
            body: JSON.stringify({
                repo_handle: repoHandle,
                is_public: isPublic,
                description,
            }),
        });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        return json;
    }
    async listCommits(repoFullName, options) {
        const { limit = 100, offset = 0 } = options ?? {};
        const res = await fetch(`${this.apiUrl}/commits/${repoFullName}/?limit=${limit}&offset=${offset}`, { headers: this._getHeaders() });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        return json;
    }
    async likeRepo(repoFullName) {
        const res = await fetch(`${this.apiUrl}/likes/${repoFullName}`, {
            method: "POST",
            headers: this._getHeaders("POST"),
            body: JSON.stringify({ like: true }),
        });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        return json;
    }
    async unlikeRepo(repoFullName) {
        const res = await fetch(`${this.apiUrl}/likes/${repoFullName}`, {
            method: "POST",
            headers: this._getHeaders("POST"),
            body: JSON.stringify({ like: false }),
        });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        return json;
    }
    async _getLatestCommitHash(repoFullName) {
        const commitsResp = await this.listCommits(repoFullName);
        const commits = commitsResp.commits;
        if (commits.length === 0) {
            return undefined;
        }
        return commits[0].commit_hash;
    }
    async updateRepo(repoFullName, options) {
        const res = await fetch(`${this.apiUrl}/repos/${repoFullName}`, {
            method: "PATCH",
            headers: this._getHeaders("PATCH"),
            body: JSON.stringify(options),
        });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        return json;
    }
    async push(repoFullName, manifestJson, options) {
        const { parentCommitHash = "latest", newRepoIsPublic = true, newRepoDescription = "", } = options ?? {};
        let repo;
        try {
            repo = await this.getRepo(repoFullName);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
        }
        catch (e) {
            if (e.statusCode !== 404) {
                throw e;
            }
            const info = parseOwnerRepoCommit(repoFullName);
            const owner = info[0];
            repo = info[1];
            const res = await this.getSettings();
            if (res.tenant_handle !== owner) {
                throw new Error(`Tenant "${res.tenant_handle}" is not the owner of repo "${repoFullName}".`);
            }
            await this.createRepo(repo, {
                description: newRepoDescription,
                isPublic: newRepoIsPublic,
            });
        }
        const resolvedParentCommitHash = parentCommitHash === "latest"
            ? await this._getLatestCommitHash(repoFullName)
            : parentCommitHash;
        const res = await fetch(`${this.apiUrl}/commits/${repoFullName}`, {
            method: "POST",
            headers: this._getHeaders("POST"),
            body: JSON.stringify({
                parent_commit: resolvedParentCommitHash,
                manifest: JSON.parse(manifestJson),
            }),
        });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        const commitHash = json.commit?.commit_hash;
        const shortHash = commitHash.slice(0, 8);
        return `${this._hostUrl}/hub/${repoFullName}/${shortHash}`;
    }
    async pull(ownerRepoCommit) {
        const info = parseOwnerRepoCommit(ownerRepoCommit);
        const [owner, repo] = info;
        let commitHash = info[2];
        if (commitHash === undefined || commitHash === "latest") {
            commitHash = await this._getLatestCommitHash(`${owner}/${repo}`);
            if (commitHash === undefined) {
                throw new Error("No commits found.");
            }
        }
        const res = await fetch(`${this.apiUrl}/commits/${owner}/${repo}/${commitHash}`, {
            method: "GET",
            headers: this._getHeaders(),
        });
        const json = await res.json();
        if (!res.ok) {
            const detail = typeof json.detail === "string"
                ? json.detail
                : JSON.stringify(json.detail);
            const error = new Error(`Error ${res.status}: ${res.statusText}\n${detail}`);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            error.statusCode = res.status;
            throw error;
        }
        const responseBody = await json;
        return JSON.stringify(responseBody.manifest);
    }
}
