/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import * as api from '@opentelemetry/api';
import { ExportResultCode, globalErrorHandler, unrefTimer } from '@opentelemetry/core';
import { MetricReader } from './MetricReader';
import { callWithTimeout, TimeoutError } from '../utils';
/**
 * {@link MetricReader} which collects metrics based on a user-configurable time interval, and passes the metrics to
 * the configured {@link MetricExporter}
 */
export class PeriodicExportingMetricReader extends MetricReader {
    constructor(options) {
        var _a, _b;
        super();
        if (options.exportIntervalMillis !== undefined && options.exportIntervalMillis <= 0) {
            throw Error('exportIntervalMillis must be greater than 0');
        }
        if (options.exportTimeoutMillis !== undefined && options.exportTimeoutMillis <= 0) {
            throw Error('exportTimeoutMillis must be greater than 0');
        }
        if (options.exportTimeoutMillis !== undefined &&
            options.exportIntervalMillis !== undefined &&
            options.exportIntervalMillis < options.exportTimeoutMillis) {
            throw Error('exportIntervalMillis must be greater than or equal to exportTimeoutMillis');
        }
        this._exportInterval = (_a = options.exportIntervalMillis) !== null && _a !== void 0 ? _a : 60000;
        this._exportTimeout = (_b = options.exportTimeoutMillis) !== null && _b !== void 0 ? _b : 30000;
        this._exporter = options.exporter;
    }
    async _runOnce() {
        const { resourceMetrics, errors } = await this.collect({});
        if (errors.length > 0) {
            api.diag.error('PeriodicExportingMetricReader: metrics collection errors', ...errors);
        }
        return new Promise((resolve, reject) => {
            this._exporter.export(resourceMetrics, result => {
                var _a;
                if (result.code !== ExportResultCode.SUCCESS) {
                    reject((_a = result.error) !== null && _a !== void 0 ? _a : new Error(`PeriodicExportingMetricReader: metrics export failed (error ${result.error})`));
                }
                else {
                    resolve();
                }
            });
        });
    }
    onInitialized() {
        // start running the interval as soon as this reader is initialized and keep handle for shutdown.
        this._interval = setInterval(async () => {
            try {
                await callWithTimeout(this._runOnce(), this._exportTimeout);
            }
            catch (err) {
                if (err instanceof TimeoutError) {
                    api.diag.error('Export took longer than %s milliseconds and timed out.', this._exportTimeout);
                    return;
                }
                globalErrorHandler(err);
            }
        }, this._exportInterval);
        unrefTimer(this._interval);
    }
    async onForceFlush() {
        await this._exporter.forceFlush();
    }
    async onShutdown() {
        if (this._interval) {
            clearInterval(this._interval);
        }
        await this._exporter.shutdown();
    }
    /**
     * @inheritdoc
     */
    selectAggregationTemporality(instrumentType) {
        return this._exporter.selectAggregationTemporality(instrumentType);
    }
}
//# sourceMappingURL=PeriodicExportingMetricReader.js.map