"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VegaMapView = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _i18n = require("@kbn/i18n");
var _vega = require("vega");
var _mapboxGl = require("@kbn/mapbox-gl");
var _layers = require("./layers");
var _vega_base_view = require("../vega_base_view");
var _services = require("../../services");
var _constants = require("./constants");
var _utils = require("./utils");
require("./vega_map_view.scss");
var _service_settings_types = require("./service_settings/service_settings_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

async function updateVegaView(mapBoxInstance, vegaView) {
  const mapCanvas = mapBoxInstance.getCanvas();
  const {
    lat,
    lng
  } = mapBoxInstance.getCenter();
  let shouldRender = false;
  const sendSignal = (sig, value) => {
    if (vegaView.signal(sig) !== value) {
      vegaView.signal(sig, value);
      shouldRender = true;
    }
  };
  sendSignal('width', mapCanvas.clientWidth);
  sendSignal('height', mapCanvas.clientHeight);
  sendSignal('latitude', lat);
  sendSignal('longitude', lng);
  sendSignal('zoom', mapBoxInstance.getZoom());
  if (shouldRender) {
    await vegaView.runAsync();
  }
}
(0, _vega.expressionFunction)('setMapView', function handlerFwd(...args) {
  const view = this.context.dataflow;
  if (!('setMapViewHandler' in view._kibanaView)) {
    // not a map view, don't do anything
    return;
  }
  view.runAfter(() => view._kibanaView.setMapViewHandler(...args));
});
class VegaMapView extends _vega_base_view.VegaBaseView {
  constructor(...args) {
    super(...args);
    (0, _defineProperty2.default)(this, "mapBoxInstance", void 0);
  }
  get shouldShowZoomControl() {
    return Boolean(this._parser.mapConfig.zoomControl);
  }
  getMapParams(defaults) {
    var _signals$zoom, _signals$longitude, _signals$latitude;
    const {
      longitude,
      latitude,
      scrollWheelZoom
    } = this._parser.mapConfig;
    const {
      zoom,
      maxZoom,
      minZoom
    } = (0, _utils.validateZoomSettings)(this._parser.mapConfig, defaults, this.onWarn.bind(this));
    const {
      signals
    } = this._vegaStateRestorer.restore() || {};
    return {
      maxZoom,
      minZoom,
      zoom: (_signals$zoom = signals === null || signals === void 0 ? void 0 : signals.zoom) !== null && _signals$zoom !== void 0 ? _signals$zoom : zoom,
      center: [(_signals$longitude = signals === null || signals === void 0 ? void 0 : signals.longitude) !== null && _signals$longitude !== void 0 ? _signals$longitude : longitude, (_signals$latitude = signals === null || signals === void 0 ? void 0 : signals.latitude) !== null && _signals$latitude !== void 0 ? _signals$latitude : latitude],
      scrollZoom: scrollWheelZoom
    };
  }
  async getEmsTileLayerId() {
    const {
      mapStyle,
      emsTileServiceId
    } = this._parser.mapConfig;
    //
    if (mapStyle) {
      const isDarkMode = (0, _services.getThemeService)().getTheme().darkMode;
      return emsTileServiceId ? emsTileServiceId : await this._serviceSettings.getDefaultTmsLayer(isDarkMode);
    }
  }
  async initMapContainer(vegaView) {
    let style = _constants.defaultMabBoxStyle;
    let customAttribution = [];
    const zoomSettings = {
      minZoom: _constants.defaultMapConfig.minZoom,
      maxZoom: _constants.defaultMapConfig.maxZoom
    };
    const emsTileLayer = await this.getEmsTileLayerId();
    if (emsTileLayer && emsTileLayer !== _service_settings_types.TMS_IN_YML_ID) {
      const tmsService = await this._serviceSettings.getTmsService(emsTileLayer);
      if (!tmsService) {
        this.onWarn(_i18n.i18n.translate('visTypeVega.mapView.mapStyleNotFoundWarningMessage', {
          defaultMessage: '{mapStyleParam} was not found',
          values: {
            mapStyleParam: `"emsTileServiceId":${emsTileLayer}`
          }
        }));
        return;
      }
      zoomSettings.maxZoom = _constants.defaultMapConfig.maxZoom;
      zoomSettings.minZoom = _constants.defaultMapConfig.minZoom;
      customAttribution = this._serviceSettings.getAttributionsFromTMSServce(tmsService);
      style = await tmsService.getVectorStyleSheet();
    } else {
      const config = this._serviceSettings.getTileMapConfig();
      customAttribution = config.options.attribution;
    }

    // In some cases, Vega may be initialized twice, e.g. after awaiting...
    if (!this._$container) return;

    // For the correct geration of the PDF/PNG report, we must wait until the map is fully rendered.
    return new Promise(resolve => {
      const mapBoxInstance = new _mapboxGl.maplibregl.Map({
        style,
        customAttribution,
        container: this._$container.get(0),
        ...this.getMapParams({
          ...zoomSettings
        })
      });
      const initMapComponents = () => {
        this.initControls(mapBoxInstance);
        this.initLayers(mapBoxInstance, vegaView, emsTileLayer);
        this._addDestroyHandler(() => {
          if (mapBoxInstance.getLayer(_constants.vegaLayerId)) {
            mapBoxInstance.removeLayer(_constants.vegaLayerId);
          }
          if (mapBoxInstance.getLayer(_service_settings_types.TMS_IN_YML_ID)) {
            mapBoxInstance.removeLayer(_service_settings_types.TMS_IN_YML_ID);
          }
          mapBoxInstance.remove();
        });
        resolve(mapBoxInstance);
      };
      mapBoxInstance.once('load', initMapComponents);
      this.mapBoxInstance = mapBoxInstance;
    });
  }
  initControls(mapBoxInstance) {
    if (this.shouldShowZoomControl) {
      mapBoxInstance.addControl(new _mapboxGl.maplibregl.NavigationControl({
        showCompass: false
      }), 'top-left');
    }

    // disable map rotation using right click + drag
    mapBoxInstance.dragRotate.disable();

    // disable map rotation using touch rotation gesture
    mapBoxInstance.touchZoomRotate.disableRotation();
  }
  initLayers(mapBoxInstance, vegaView, emsTileLayer) {
    var _this$_$controls;
    const shouldShowUserConfiguredLayer = emsTileLayer === _service_settings_types.TMS_IN_YML_ID;
    if (shouldShowUserConfiguredLayer) {
      var _options$maxZoom, _options$minZoom, _options$tileSize;
      const {
        url,
        options
      } = this._serviceSettings.getTileMapConfig();
      (0, _layers.initTmsRasterLayer)({
        id: _service_settings_types.TMS_IN_YML_ID,
        map: mapBoxInstance,
        context: {
          tiles: [url],
          maxZoom: (_options$maxZoom = options.maxZoom) !== null && _options$maxZoom !== void 0 ? _options$maxZoom : _constants.defaultMapConfig.maxZoom,
          minZoom: (_options$minZoom = options.minZoom) !== null && _options$minZoom !== void 0 ? _options$minZoom : _constants.defaultMapConfig.minZoom,
          tileSize: (_options$tileSize = options.tileSize) !== null && _options$tileSize !== void 0 ? _options$tileSize : _constants.defaultMapConfig.tileSize
        }
      });
    }
    (0, _layers.initVegaLayer)({
      id: _constants.vegaLayerId,
      map: mapBoxInstance,
      context: {
        vegaView,
        vegaControls: (_this$_$controls = this._$controls) === null || _this$_$controls === void 0 ? void 0 : _this$_$controls.get(0),
        updateVegaView
      }
    });
  }
  async _initViewCustomizations() {
    const vegaView = new _vega.View((0, _vega.parse)((0, _utils.injectMapPropsIntoSpec)(this._parser.spec), undefined, {
      ast: true
    }), this._vegaViewConfig);
    this.setDebugValues(vegaView, this._parser.spec, this._parser.vlspec);
    this.setView(vegaView);
    await this.initMapContainer(vegaView);
  }
  async onViewContainerResize() {
    var _this$mapBoxInstance;
    (_this$mapBoxInstance = this.mapBoxInstance) === null || _this$mapBoxInstance === void 0 ? void 0 : _this$mapBoxInstance.resize();
  }
  setMapViewHandler(...args) {
    if (!this.mapBoxInstance) {
      return;
    }
    function throwError() {
      throw new Error(_i18n.i18n.translate('visTypeVega.visualization.setMapViewErrorMessage', {
        defaultMessage: 'Unexpected setMapView() parameters. It could be called with a bounding box setMapView([[longitude1,latitude1],[longitude2,latitude2]]), or it could be the center point setMapView([longitude, latitude], optional_zoom), or it can be used as setMapView(latitude, longitude, optional_zoom)'
      }));
    }
    function checkArray(val) {
      if (!Array.isArray(val) || val.length !== 2 || typeof val[0] !== 'number' || typeof val[1] !== 'number') {
        throwError();
      }
      return val;
    }
    let lng;
    let lat;
    let zoom;
    switch (args.length) {
      default:
        throwError();
        break;
      case 1:
        {
          const arg = args[0];
          if (Array.isArray(arg) && arg.length === 2 && Array.isArray(arg[0]) && Array.isArray(arg[1])) {
            // called with a bounding box, need to reverse order
            const [lng1, lat1] = checkArray(arg[0]);
            const [lng2, lat2] = checkArray(arg[1]);
            this.mapBoxInstance.fitBounds([{
              lat: lat1,
              lng: lng1
            }, {
              lat: lat2,
              lng: lng2
            }]);
          } else {
            // called with a center point and no zoom
            [lng, lat] = checkArray(arg);
          }
          break;
        }
      case 2:
        if (Array.isArray(args[0])) {
          [lng, lat] = checkArray(args[0]);
          zoom = args[1];
        } else {
          [lat, lng] = args;
        }
        break;
      case 3:
        [lat, lng, zoom] = args;
        break;
    }
    if (lat !== undefined && lng !== undefined) {
      if (typeof lat !== 'number' || typeof lng !== 'number') {
        throwError();
      }
      if (zoom !== undefined && typeof zoom !== 'number') {
        throwError();
      }
      this.mapBoxInstance.setCenter({
        lat,
        lng
      });
      if (zoom !== undefined) {
        this.mapBoxInstance.zoomTo(zoom);
      }
    }
  }
}
exports.VegaMapView = VegaMapView;