"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useUiState = void 0;
var _lodash = require("lodash");
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const defaultSort = {
  columnIndex: null,
  direction: null
};
const useUiState = uiState => {
  const [sort, setSortState] = (0, _react.useState)((uiState === null || uiState === void 0 ? void 0 : uiState.get('vis.params.sort')) || defaultSort);
  const [columnsWidth, setColumnsWidthState] = (0, _react.useState)((uiState === null || uiState === void 0 ? void 0 : uiState.get('vis.params.colWidth')) || []);
  const uiStateValues = (0, _react.useRef)({
    columnsWidth: uiState === null || uiState === void 0 ? void 0 : uiState.get('vis.params.colWidth'),
    sort: uiState === null || uiState === void 0 ? void 0 : uiState.get('vis.params.sort'),
    pendingUpdate: false
  });
  const setSort = (0, _react.useCallback)((s = defaultSort) => {
    setSortState(s || defaultSort);
    uiStateValues.current.sort = s;
    uiStateValues.current.pendingUpdate = true;

    /**
     * Since the visualize app state is listening for uiState changes,
     * it synchronously re-renders an editor frame.
     * Setting new uiState values in the new event loop task,
     * helps to update the visualization frame firstly and not to block the rendering flow
     */
    setTimeout(() => {
      uiState === null || uiState === void 0 ? void 0 : uiState.set('vis.params.sort', s);
      uiStateValues.current.pendingUpdate = false;
    });
  }, [uiState]);
  const setColumnsWidth = (0, _react.useCallback)(col => {
    setColumnsWidthState(prevState => {
      const updated = [...prevState];
      const idx = prevState.findIndex(c => c.colIndex === col.colIndex);
      if (idx >= 0) {
        updated[idx] = col;
      } else {
        updated.push(col);
      }
      uiStateValues.current.columnsWidth = updated;
      uiStateValues.current.pendingUpdate = true;

      /**
       * Since the visualize app state is listening for uiState changes,
       * it synchronously re-renders an editor frame.
       * Setting new uiState values in the new event loop task,
       * helps to update the visualization frame firstly and not to block the rendering flow
       */
      setTimeout(() => {
        uiState === null || uiState === void 0 ? void 0 : uiState.set('vis.params.colWidth', updated);
        uiStateValues.current.pendingUpdate = false;
      });
      return updated;
    });
  }, [uiState]);
  (0, _react.useEffect)(() => {
    /**
     * Debounce is in place since there are couple of synchronous updates of the uiState,
     * which are also handled synchronously.
     */
    const updateOnChange = (0, _lodash.debounce)(() => {
      // skip uiState updates if there are pending internal state updates
      if (uiStateValues.current.pendingUpdate) {
        return;
      }
      const {
        vis
      } = uiState === null || uiState === void 0 ? void 0 : uiState.getChanges();
      if (!(0, _lodash.isEqual)(vis === null || vis === void 0 ? void 0 : vis.params.colWidth, uiStateValues.current.columnsWidth)) {
        uiStateValues.current.columnsWidth = vis === null || vis === void 0 ? void 0 : vis.params.colWidth;
        setColumnsWidthState((vis === null || vis === void 0 ? void 0 : vis.params.colWidth) || []);
      }
      if (!(0, _lodash.isEqual)(vis === null || vis === void 0 ? void 0 : vis.params.sort, uiStateValues.current.sort)) {
        uiStateValues.current.sort = vis === null || vis === void 0 ? void 0 : vis.params.sort;
        setSortState((vis === null || vis === void 0 ? void 0 : vis.params.sort) || defaultSort);
      }
    });
    uiState === null || uiState === void 0 ? void 0 : uiState.on('change', updateOnChange);
    return () => {
      uiState === null || uiState === void 0 ? void 0 : uiState.off('change', updateOnChange);
    };
  }, [uiState]);
  return {
    columnsWidth,
    sort,
    setColumnsWidth,
    setSort
  };
};
exports.useUiState = useUiState;