"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initUptimeServer = void 0;
var _coreHttpServer = require("@kbn/core-http-server");
var _constants = require("../../common/constants");
var _dynamic_settings = require("./routes/dynamic_settings");
var _requests = require("./lib/requests");
var _routes = require("./routes");
var _status_check = require("./lib/alerts/status_check");
var _tls = require("./lib/alerts/tls");
var _tls_legacy = require("./lib/alerts/tls_legacy");
var _duration_anomaly = require("./lib/alerts/duration_anomaly");
var _domains = require("./lib/domains");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const libs = {
  requests: _requests.uptimeRequests,
  license: _domains.licenseCheck
};
const initUptimeServer = (server, plugins, ruleDataClient, logger, router) => {
  _routes.legacyUptimeRestApiRoutes.forEach(route => {
    const {
      method,
      options,
      handler,
      validate,
      path
    } = (0, _routes.uptimeRouteWrapper)((0, _routes.createRouteWithAuth)(libs, route), server);
    const routeDefinition = {
      path,
      validate,
      options
    };
    switch (method) {
      case 'GET':
        router.get(routeDefinition, handler);
        break;
      case 'POST':
        router.post(routeDefinition, handler);
        break;
      case 'PUT':
        router.put(routeDefinition, handler);
        break;
      case 'DELETE':
        router.delete(routeDefinition, handler);
        break;
      default:
        throw new Error(`Handler for method ${method} is not defined`);
    }
  });
  _routes.legacyUptimePublicRestApiRoutes.forEach(route => {
    const {
      method,
      options,
      handler,
      path,
      ...rest
    } = (0, _routes.uptimeRouteWrapper)((0, _routes.createRouteWithAuth)(libs, route), server);
    const validate = rest.validate ? (0, _coreHttpServer.getRequestValidation)(rest.validate) : rest.validate;
    const routeDefinition = {
      path,
      validate,
      options
    };
    switch (method) {
      case 'GET':
        router.versioned.get({
          access: 'public',
          path: routeDefinition.path,
          options: {
            tags: options === null || options === void 0 ? void 0 : options.tags
          }
        }).addVersion({
          version: _constants.INITIAL_REST_VERSION,
          validate: {
            request: {
              body: validate ? validate === null || validate === void 0 ? void 0 : validate.body : undefined
            },
            response: {
              200: {
                body: () => _dynamic_settings.DynamicSettingsSchema
              }
            }
          }
        }, handler);
        break;
      case 'PUT':
        router.versioned.put({
          access: 'public',
          path: routeDefinition.path,
          options: {
            tags: options === null || options === void 0 ? void 0 : options.tags
          }
        }).addVersion({
          version: _constants.INITIAL_REST_VERSION,
          validate: {
            request: {
              body: validate ? validate === null || validate === void 0 ? void 0 : validate.body : undefined
            },
            response: {
              200: {
                body: () => _dynamic_settings.DynamicSettingsSchema
              }
            }
          }
        }, handler);
        break;
      default:
        throw new Error(`Handler for method ${method} is not defined`);
    }
  });
  const {
    alerting: {
      registerType
    }
  } = plugins;
  const statusAlert = (0, _status_check.statusCheckAlertFactory)(server, libs, plugins);
  const tlsLegacyRule = (0, _tls_legacy.tlsLegacyRuleFactory)(server, libs, plugins);
  const tlsAlert = (0, _tls.tlsAlertFactory)(server, libs, plugins);
  const durationAlert = (0, _duration_anomaly.durationAnomalyAlertFactory)(server, libs, plugins);
  registerType(statusAlert);
  registerType(tlsAlert);
  registerType(durationAlert);

  /* TLS Legacy rule supported at least through 8.0.
   * Not registered with RAC */
  registerType(tlsLegacyRule);
};
exports.initUptimeServer = initUptimeServer;