"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLoadRuleTypesQuery = void 0;
var _i18n = require("@kbn/i18n");
var _reactQuery = require("@tanstack/react-query");
var _common = require("@kbn/alerting-plugin/common");
var _react = require("react");
var _rule_types = require("../lib/rule_api/rule_types");
var _kibana = require("../../common/lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getFilteredIndex = (data, filteredRuleTypes) => {
  const index = new Map();
  for (const ruleType of data) {
    index.set(ruleType.id, ruleType);
  }
  let filteredIndex = index;
  if (filteredRuleTypes !== null && filteredRuleTypes !== void 0 && filteredRuleTypes.length) {
    filteredIndex = new Map([...index].filter(([k, v]) => {
      return filteredRuleTypes.includes(v.id);
    }));
  }
  return filteredIndex;
};
const useLoadRuleTypesQuery = ({
  filteredRuleTypes,
  enabled = true
}) => {
  const {
    http,
    notifications: {
      toasts
    }
  } = (0, _kibana.useKibana)().services;
  const queryFn = () => {
    return (0, _rule_types.loadRuleTypes)({
      http
    });
  };
  const onErrorFn = () => {
    toasts.addDanger(_i18n.i18n.translate('xpack.triggersActionsUI.sections.rulesList.unableToLoadRuleTypesMessage', {
      defaultMessage: 'Unable to load rule types'
    }));
  };
  const {
    data,
    isSuccess,
    isFetching,
    isInitialLoading,
    isLoading,
    error
  } = (0, _reactQuery.useQuery)({
    queryKey: ['loadRuleTypes'],
    queryFn,
    onError: onErrorFn,
    refetchOnWindowFocus: false,
    // Leveraging TanStack Query's caching system to avoid duplicated requests as
    // other state-sharing solutions turned out to be overly complex and less readable
    staleTime: 60 * 1000,
    enabled
  });
  const filteredIndex = (0, _react.useMemo)(() => data ? getFilteredIndex(data, filteredRuleTypes) : new Map(), [data, filteredRuleTypes]);
  const hasAnyAuthorizedRuleType = filteredIndex.size > 0;
  const authorizedRuleTypes = (0, _react.useMemo)(() => [...filteredIndex.values()], [filteredIndex]);
  const authorizedToCreateAnyRules = authorizedRuleTypes.some(ruleType => {
    var _ruleType$authorizedC;
    return (_ruleType$authorizedC = ruleType.authorizedConsumers[_common.ALERTING_FEATURE_ID]) === null || _ruleType$authorizedC === void 0 ? void 0 : _ruleType$authorizedC.all;
  });
  const authorizedToReadAnyRules = authorizedToCreateAnyRules || authorizedRuleTypes.some(ruleType => {
    var _ruleType$authorizedC2;
    return (_ruleType$authorizedC2 = ruleType.authorizedConsumers[_common.ALERTING_FEATURE_ID]) === null || _ruleType$authorizedC2 === void 0 ? void 0 : _ruleType$authorizedC2.read;
  });
  return {
    ruleTypesState: {
      initialLoad: isLoading || isInitialLoading,
      isLoading: isLoading || isFetching,
      data: filteredIndex,
      error
    },
    hasAnyAuthorizedRuleType,
    authorizedRuleTypes,
    authorizedToReadAnyRules,
    authorizedToCreateAnyRules,
    isSuccess
  };
};
exports.useLoadRuleTypesQuery = useLoadRuleTypesQuery;