"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.timelineSearchStrategyProvider = void 0;
var _rxjs = require("rxjs");
var _server = require("@kbn/data-plugin/server");
var _common = require("@kbn/data-plugin/common");
var _search_strategy = require("../../../common/api/search_strategy");
var _timeline = require("../../../common/search_strategy/timeline");
var _factory = require("./factory");
var _is_agg_cardinality_aggregate = require("./factory/helpers/is_agg_cardinality_aggregate");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const timelineSearchStrategyProvider = (data, logger, security) => {
  const es = data.search.getSearchStrategy(_common.ENHANCED_ES_SEARCH_STRATEGY);
  return {
    search: (request, options, deps) => {
      const entityType = request.entityType;
      const searchStrategyRequest = _search_strategy.searchStrategyRequestSchema.parse(request);
      const queryFactory = _factory.timelineFactory[searchStrategyRequest.factoryQueryType];
      if (entityType != null && entityType === _timeline.EntityType.SESSIONS) {
        return timelineSessionsSearchStrategy({
          es,
          request: searchStrategyRequest,
          options,
          deps,
          queryFactory
        });
      } else {
        return timelineSearchStrategy({
          es,
          request: searchStrategyRequest,
          options,
          deps,
          queryFactory,
          logger
        });
      }
    },
    cancel: async (id, options, deps) => {
      if (es.cancel) {
        return es.cancel(id, options, deps);
      }
    }
  };
};
exports.timelineSearchStrategyProvider = timelineSearchStrategyProvider;
const timelineSearchStrategy = ({
  es,
  request,
  options,
  deps,
  queryFactory
}) => {
  const dsl = queryFactory.buildDsl(request);
  return es.search({
    ...request,
    params: dsl
  }, options, deps).pipe((0, _rxjs.map)(response => {
    return {
      ...response,
      rawResponse: (0, _server.shimHitsTotal)(response.rawResponse, options)
    };
  }), (0, _rxjs.mergeMap)(esSearchRes => queryFactory.parse(request, esSearchRes)));
};
const timelineSessionsSearchStrategy = ({
  es,
  request,
  options,
  deps,
  queryFactory
}) => {
  var _request$defaultIndex;
  const indices = (_request$defaultIndex = request.defaultIndex) !== null && _request$defaultIndex !== void 0 ? _request$defaultIndex : request.indexType;
  const requestSessionLeaders = {
    ...request,
    defaultIndex: indices,
    indexName: indices
  };
  const collapse = {
    field: 'process.entry_leader.entity_id'
  };
  const aggs = {
    total: {
      cardinality: {
        field: 'process.entry_leader.entity_id'
      }
    }
  };
  const dsl = queryFactory.buildDsl(requestSessionLeaders);
  const params = {
    ...dsl,
    collapse,
    aggs
  };
  return es.search({
    ...requestSessionLeaders,
    params
  }, options, deps).pipe((0, _rxjs.map)(response => {
    const agg = response.rawResponse.aggregations;
    const aggTotal = (0, _is_agg_cardinality_aggregate.isAggCardinalityAggregate)(agg, 'total') && agg.total.value;

    // ES doesn't set the hits.total to the collapsed hits.
    // so we are overriding hits.total with the total from the aggregation.
    if (aggTotal) {
      response.rawResponse.hits.total = aggTotal;
    }
    return {
      ...response,
      rawResponse: (0, _server.shimHitsTotal)(response.rawResponse, options)
    };
  }), (0, _rxjs.mergeMap)(esSearchRes => queryFactory.parse(requestSessionLeaders, esSearchRes)));
};