"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addPrivateLocationRoute = exports.PrivateLocationSchema = void 0;
var _configSchema = require("@kbn/config-schema");
var _get_private_locations = require("./get_private_locations");
var _private_locations = require("../../../../common/saved_objects/private_locations");
var _constants = require("../../../../common/constants");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PrivateLocationSchema = exports.PrivateLocationSchema = _configSchema.schema.object({
  label: _configSchema.schema.string(),
  agentPolicyId: _configSchema.schema.string(),
  tags: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
  geo: _configSchema.schema.maybe(_configSchema.schema.object({
    lat: _configSchema.schema.number(),
    lon: _configSchema.schema.number()
  }))
});
const addPrivateLocationRoute = () => ({
  method: 'POST',
  path: _constants.SYNTHETICS_API_URLS.PRIVATE_LOCATIONS,
  validate: {},
  validation: {
    request: {
      body: PrivateLocationSchema
    }
  },
  handler: async ({
    response,
    request,
    savedObjectsClient,
    syntheticsMonitorClient
  }) => {
    const location = request.body;
    const {
      locations,
      agentPolicies
    } = await (0, _get_private_locations.getPrivateLocationsAndAgentPolicies)(savedObjectsClient, syntheticsMonitorClient);
    if (locations.find(loc => loc.agentPolicyId === location.agentPolicyId)) {
      return response.badRequest({
        body: {
          message: `Private location with agentPolicyId ${location.agentPolicyId} already exists`
        }
      });
    }

    // return if name is already taken
    if (locations.find(loc => loc.label === location.label)) {
      return response.badRequest({
        body: {
          message: `Private location with label ${location.label} already exists`
        }
      });
    }
    const existingLocations = locations.filter(loc => loc.id !== location.agentPolicyId);
    const formattedLocation = (0, _helpers.toSavedObjectContract)({
      ...location,
      id: location.agentPolicyId
    });
    const agentPolicy = agentPolicies === null || agentPolicies === void 0 ? void 0 : agentPolicies.find(policy => policy.id === location.agentPolicyId);
    if (!agentPolicy) {
      return response.badRequest({
        body: {
          message: `Agent policy with id ${location.agentPolicyId} does not exist`
        }
      });
    }
    const result = await savedObjectsClient.create(_private_locations.privateLocationsSavedObjectName, {
      locations: [...existingLocations, formattedLocation]
    }, {
      id: _private_locations.privateLocationsSavedObjectId,
      overwrite: true
    });
    const allLocations = (0, _helpers.toClientContract)(result.attributes, agentPolicies);
    return allLocations.find(loc => loc.id === location.agentPolicyId);
  }
});
exports.addPrivateLocationRoute = addPrivateLocationRoute;