"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.invalidOriginError = exports.addSyntheticsMonitorRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _i18n = require("@kbn/i18n");
var _common_fields = require("../../synthetics_service/project_monitor/normalizers/common_fields");
var _add_monitor_api = require("./add_monitor/add_monitor_api");
var _constants = require("../../../common/constants");
var _monitor_validation = require("./monitor_validation");
var _helper = require("./helper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const addSyntheticsMonitorRoute = () => ({
  method: 'POST',
  path: _constants.SYNTHETICS_API_URLS.SYNTHETICS_MONITORS,
  validate: {},
  validation: {
    request: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        id: _configSchema.schema.maybe(_configSchema.schema.string()),
        preserve_namespace: _configSchema.schema.maybe(_configSchema.schema.boolean()),
        gettingStarted: _configSchema.schema.maybe(_configSchema.schema.boolean())
      })
    }
  },
  handler: async routeContext => {
    const {
      request,
      response,
      server
    } = routeContext;
    // usually id is auto generated, but this is useful for testing
    const {
      id
    } = request.query;
    const addMonitorAPI = new _add_monitor_api.AddEditMonitorAPI(routeContext);
    const {
      locations,
      private_locations: privateLocations,
      ...monitor
    } = request.body;
    if (request.body.origin && request.body.origin !== 'ui') {
      return response.badRequest({
        body: {
          message: invalidOriginError(request.body.origin),
          attributes: {
            details: invalidOriginError(request.body.origin),
            payload: request.body
          }
        }
      });
    }
    try {
      const {
        errorMessage: unsupportedKeysErrors,
        formattedConfig
      } = (0, _monitor_validation.normalizeAPIConfig)(request.body);
      if (unsupportedKeysErrors) {
        return response.badRequest({
          body: {
            message: unsupportedKeysErrors,
            attributes: {
              details: unsupportedKeysErrors
            }
          }
        });
      }
      const monitorWithDefaults = await addMonitorAPI.normalizeMonitor(formattedConfig, request.body);
      const validationResult = (0, _monitor_validation.validateMonitor)(monitorWithDefaults);
      if (!validationResult.valid || !validationResult.decodedMonitor) {
        const {
          reason: message,
          details
        } = validationResult;
        return response.badRequest({
          body: {
            message,
            attributes: {
              details,
              payload: monitor
            }
          }
        });
      }
      const normalizedMonitor = validationResult.decodedMonitor;
      const nameError = await addMonitorAPI.validateUniqueMonitorName(normalizedMonitor.name);
      if (nameError) {
        return response.badRequest({
          body: {
            message: nameError,
            attributes: {
              details: nameError
            }
          }
        });
      }
      const {
        errors,
        newMonitor
      } = await addMonitorAPI.syncNewMonitor({
        id,
        normalizedMonitor
      });
      if (errors && errors.length > 0) {
        return {
          message: 'error pushing monitor to the service',
          attributes: {
            errors
          },
          id: newMonitor.id
        };
      }
      addMonitorAPI.initDefaultAlerts(newMonitor.attributes.name);
      addMonitorAPI.setupGettingStarted(newMonitor.id);
      return (0, _helper.mapSavedObjectToMonitor)(newMonitor);
    } catch (getErr) {
      server.logger.error(getErr);
      if (getErr instanceof _common_fields.InvalidLocationError) {
        return response.badRequest({
          body: {
            message: getErr.message
          }
        });
      }
      if (_server.SavedObjectsErrorHelpers.isForbiddenError(getErr)) {
        return response.forbidden({
          body: getErr
        });
      }
      return response.customError({
        body: {
          message: getErr.message
        },
        statusCode: 500
      });
    }
  }
});
exports.addSyntheticsMonitorRoute = addSyntheticsMonitorRoute;
const invalidOriginError = origin => {
  return _i18n.i18n.translate('xpack.synthetics.server.projectMonitors.invalidPublicOriginError', {
    defaultMessage: 'Unsupported origin type {origin}, only ui type is supported via API.',
    values: {
      origin
    }
  });
};
exports.invalidOriginError = invalidOriginError;