"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CrowdstrikeConnector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/actions-plugin/server");
var _schema = require("../../../common/crowdstrike/schema");
var _constants = require("../../../common/crowdstrike/constants");
var _error = require("./error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const paramsSerializer = params => {
  return Object.entries(params).map(([key, value]) => `${encodeURIComponent(key)}=${encodeURIComponent(value)}`).join('&');
};

/**
 * Crowdstrike Connector
 * @constructor
 * @param {string} token - Authorization token received from OAuth2 API, that needs to be sent along with each request.
 * @param {number} tokenExpiryTimeout - Tokens are valid for 30 minutes, so we will refresh them every 29 minutes
 * @param {base64} base64encodedToken - The base64 encoded token used for authentication.
 */

class CrowdstrikeConnector extends _server.SubActionConnector {
  constructor(params) {
    super(params);
    (0, _defineProperty2.default)(this, "urls", void 0);
    this.urls = {
      getToken: `${this.config.url}/oauth2/token`,
      hostAction: `${this.config.url}/devices/entities/devices-actions/v2`,
      agents: `${this.config.url}/devices/entities/devices/v2`
    };
    if (!CrowdstrikeConnector.base64encodedToken) {
      CrowdstrikeConnector.base64encodedToken = Buffer.from(this.secrets.clientId + ':' + this.secrets.clientSecret).toString('base64');
    }
    this.registerSubActions();
  }
  registerSubActions() {
    this.registerSubAction({
      name: _constants.SUB_ACTION.GET_AGENT_DETAILS,
      method: 'getAgentDetails',
      schema: _schema.CrowdstrikeGetAgentsParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.HOST_ACTIONS,
      method: 'executeHostActions',
      schema: _schema.CrowdstrikeHostActionsParamsSchema
    });
  }
  async executeHostActions({
    alertIds,
    ...payload
  }) {
    return this.crowdstrikeApiRequest({
      url: this.urls.hostAction,
      method: 'post',
      params: {
        action_name: payload.command
      },
      data: {
        ids: payload.ids,
        ...(payload.actionParameters ? {
          action_parameters: Object.entries(payload.actionParameters).map(([name, value]) => ({
            name,
            value
          }))
        } : {})
      },
      paramsSerializer,
      responseSchema: _schema.CrowdstrikeHostActionsResponseSchema
    });
  }
  async getAgentDetails(payload) {
    return this.crowdstrikeApiRequest({
      url: this.urls.agents,
      method: 'GET',
      params: {
        ids: payload.ids
      },
      paramsSerializer,
      responseSchema: _schema.CrowdstrikeGetAgentsResponseSchema
    });
  }
  async getTokenRequest() {
    var _response$data;
    const response = await this.request({
      url: this.urls.getToken,
      method: 'post',
      headers: {
        accept: 'application/json',
        'Content-Type': 'application/x-www-form-urlencoded',
        authorization: 'Basic ' + this.base64encodedToken
      },
      responseSchema: _schema.CrowdstrikeGetTokenResponseSchema
    });
    const token = (_response$data = response.data) === null || _response$data === void 0 ? void 0 : _response$data.access_token;
    if (token) {
      // Clear any existing timeout
      clearTimeout(CrowdstrikeConnector.tokenExpiryTimeout);

      // Set a timeout to reset the token after 29 minutes (it expires after 30 minutes)
      CrowdstrikeConnector.tokenExpiryTimeout = setTimeout(() => {
        CrowdstrikeConnector.token = null;
      }, 29 * 60 * 1000);
    }
    return token;
  }
  async crowdstrikeApiRequest(req, retried) {
    try {
      if (!CrowdstrikeConnector.token) {
        CrowdstrikeConnector.token = await this.getTokenRequest();
      }
      const response = await this.request({
        ...req,
        headers: {
          ...req.headers,
          Authorization: `Bearer ${CrowdstrikeConnector.token}`
        }
      });
      return response.data;
    } catch (error) {
      if (error.code === 401 && !retried) {
        CrowdstrikeConnector.token = null;
        return this.crowdstrikeApiRequest(req, true);
      }
      throw new _error.CrowdstrikeError(error.message);
    }
  }
  getResponseErrorMessage(error) {
    var _error$response, _error$response$data, _error$response$data$, _error$response2, _error$response$data3;
    const errorData = (_error$response = error.response) === null || _error$response === void 0 ? void 0 : (_error$response$data = _error$response.data) === null || _error$response$data === void 0 ? void 0 : (_error$response$data$ = _error$response$data.errors) === null || _error$response$data$ === void 0 ? void 0 : _error$response$data$[0];
    if (errorData) {
      return errorData.message;
    }
    if (!((_error$response2 = error.response) !== null && _error$response2 !== void 0 && _error$response2.status)) {
      var _error$response$data2, _error$response3;
      return `Unknown API Error: ${JSON.stringify((_error$response$data2 = (_error$response3 = error.response) === null || _error$response3 === void 0 ? void 0 : _error$response3.data) !== null && _error$response$data2 !== void 0 ? _error$response$data2 : {})}`;
    }
    return `API Error: ${JSON.stringify((_error$response$data3 = error.response.data) !== null && _error$response$data3 !== void 0 ? _error$response$data3 : {})}`;
  }
}
exports.CrowdstrikeConnector = CrowdstrikeConnector;
(0, _defineProperty2.default)(CrowdstrikeConnector, "token", void 0);
(0, _defineProperty2.default)(CrowdstrikeConnector, "tokenExpiryTimeout", void 0);
(0, _defineProperty2.default)(CrowdstrikeConnector, "base64encodedToken", void 0);