"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BedrockConnector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/actions-plugin/server");
var _aws = _interopRequireDefault(require("aws4"));
var _stream = require("stream");
var _create_gen_ai_dashboard = require("../lib/gen_ai/create_gen_ai_dashboard");
var _schema = require("../../../common/bedrock/schema");
var _constants = require("../../../common/bedrock/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class BedrockConnector extends _server.SubActionConnector {
  constructor(params) {
    super(params);
    (0, _defineProperty2.default)(this, "url", void 0);
    (0, _defineProperty2.default)(this, "model", void 0);
    this.url = this.config.apiUrl;
    this.model = this.config.defaultModel;
    this.registerSubActions();
  }
  registerSubActions() {
    this.registerSubAction({
      name: _constants.SUB_ACTION.RUN,
      method: 'runApi',
      schema: _schema.RunActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.DASHBOARD,
      method: 'getDashboard',
      schema: _schema.DashboardActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.TEST,
      method: 'runApi',
      schema: _schema.RunActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.INVOKE_AI,
      method: 'invokeAI',
      schema: _schema.InvokeAIActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.INVOKE_STREAM,
      method: 'invokeStream',
      schema: _schema.InvokeAIActionParamsSchema
    });
  }
  getResponseErrorMessage(error) {
    var _error$response, _error$response2, _error$response2$data, _error$response4, _error$response5, _error$response5$data;
    if (!((_error$response = error.response) !== null && _error$response !== void 0 && _error$response.status)) {
      var _error$code, _error$message;
      return `Unexpected API Error: ${(_error$code = error.code) !== null && _error$code !== void 0 ? _error$code : ''} - ${(_error$message = error.message) !== null && _error$message !== void 0 ? _error$message : 'Unknown error'}`;
    }
    if (error.response.status === 400 && ((_error$response2 = error.response) === null || _error$response2 === void 0 ? void 0 : (_error$response2$data = _error$response2.data) === null || _error$response2$data === void 0 ? void 0 : _error$response2$data.message) === 'The requested operation is not recognized by the service.') {
      // Leave space in the string below, \n is not being rendered in the UI
      return `API Error: ${error.response.data.message}

The Kibana Connector in use may need to be reconfigured with an updated Amazon Bedrock endpoint, like \`bedrock-runtime\`.`;
    }
    if (error.response.status === 401) {
      var _error$response3, _error$response3$data;
      return `Unauthorized API Error${(_error$response3 = error.response) !== null && _error$response3 !== void 0 && (_error$response3$data = _error$response3.data) !== null && _error$response3$data !== void 0 && _error$response3$data.message ? `: ${error.response.data.message}` : ''}`;
    }
    return `API Error: ${(_error$response4 = error.response) === null || _error$response4 === void 0 ? void 0 : _error$response4.statusText}${(_error$response5 = error.response) !== null && _error$response5 !== void 0 && (_error$response5$data = _error$response5.data) !== null && _error$response5$data !== void 0 && _error$response5$data.message ? ` - ${error.response.data.message}` : ''}`;
  }

  /**
   * provides the AWS signature to the external API endpoint
   * @param body The request body to be signed.
   * @param path The path of the request URL.
   */
  signRequest(body, path, stream) {
    const {
      host
    } = new URL(this.url);
    return _aws.default.sign({
      host,
      headers: stream ? {
        accept: 'application/vnd.amazon.eventstream',
        'Content-Type': 'application/json',
        'x-amzn-bedrock-accept': '*/*'
      } : {
        'Content-Type': 'application/json',
        Accept: '*/*'
      },
      body,
      path,
      // Despite AWS docs, this value does not always get inferred. We need to always send it
      service: 'bedrock'
    }, {
      secretAccessKey: this.secrets.secret,
      accessKeyId: this.secrets.accessKey
    });
  }

  /**
   *  retrieves a dashboard from the Kibana server and checks if the
   *  user has the necessary privileges to access it.
   * @param dashboardId The ID of the dashboard to retrieve.
   */
  async getDashboard({
    dashboardId
  }) {
    const privilege = await this.esClient.transport.request({
      path: '/_security/user/_has_privileges',
      method: 'POST',
      body: {
        index: [{
          names: ['.kibana-event-log-*'],
          allow_restricted_indices: true,
          privileges: ['read']
        }]
      }
    });
    if (!(privilege !== null && privilege !== void 0 && privilege.has_all_requested)) {
      return {
        available: false
      };
    }
    const response = await (0, _create_gen_ai_dashboard.initDashboard)({
      logger: this.logger,
      savedObjectsClient: this.savedObjectsClient,
      dashboardId,
      genAIProvider: 'Bedrock'
    });
    return {
      available: response.success
    };
  }
  async runApiDeprecated(params) {
    const response = await this.request(params);
    return response.data;
  }
  async runApiLatest(params) {
    const response = await this.request(params);
    // keeping the response the same as claude 2 for our APIs
    // adding the usage object for better token tracking
    return {
      completion: parseContent(response.data.content),
      stop_reason: response.data.stop_reason,
      usage: response.data.usage
    };
  }

  /**
   * responsible for making a POST request to the external API endpoint and returning the response data
   * @param body The stringified request body to be sent in the POST request.
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   */
  async runApi({
    body,
    model: reqModel,
    signal,
    timeout
  }) {
    // set model on per request basis
    const currentModel = reqModel !== null && reqModel !== void 0 ? reqModel : this.model;
    const path = `/model/${currentModel}/invoke`;
    const signed = this.signRequest(body, path, false);
    const requestArgs = {
      ...signed,
      url: `${this.url}${path}`,
      method: 'post',
      data: body,
      signal,
      // give up to 2 minutes for response
      timeout: timeout !== null && timeout !== void 0 ? timeout : _constants.DEFAULT_TIMEOUT_MS
    };
    // possible api received deprecated arguments, which will still work with the deprecated Claude 2 models
    if (usesDeprecatedArguments(body)) {
      return this.runApiDeprecated({
        ...requestArgs,
        responseSchema: _schema.RunActionResponseSchema
      });
    }
    return this.runApiLatest({
      ...requestArgs,
      responseSchema: _schema.RunApiLatestResponseSchema
    });
  }

  /**
   *  NOT INTENDED TO BE CALLED DIRECTLY
   *  call invokeStream instead
   *  responsible for making a POST request to a specified URL with a given request body.
   *  The response is then processed based on whether it is a streaming response or a regular response.
   * @param body The stringified request body to be sent in the POST request.
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   */
  async streamApi({
    body,
    model: reqModel,
    signal,
    timeout
  }) {
    // set model on per request basis
    const path = `/model/${reqModel !== null && reqModel !== void 0 ? reqModel : this.model}/invoke-with-response-stream`;
    const signed = this.signRequest(body, path, true);
    const response = await this.request({
      ...signed,
      url: `${this.url}${path}`,
      method: 'post',
      responseSchema: _schema.StreamingResponseSchema,
      data: body,
      responseType: 'stream',
      signal,
      timeout
    });
    return response.data.pipe(new _stream.PassThrough());
  }

  /**
   *  takes in an array of messages and a model as inputs. It calls the streamApi method to make a
   *  request to the Bedrock API with the formatted messages and model. It then returns a Transform stream
   *  that pipes the response from the API through the transformToString function,
   *  which parses the proprietary response into a string of the response text alone
   * @param messages An array of messages to be sent to the API
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   */
  async invokeStream({
    messages,
    model,
    stopSequences,
    system,
    temperature,
    signal,
    timeout
  }) {
    const res = await this.streamApi({
      body: JSON.stringify(formatBedrockBody({
        messages,
        stopSequences,
        system,
        temperature
      })),
      model,
      signal,
      timeout
    });
    return res;
  }

  /**
   * Non-streamed security solution AI Assistant requests
   * Responsible for invoking the runApi method with the provided body.
   * It then formats the response into a string
   * @param messages An array of messages to be sent to the API
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   * @returns an object with the response string as a property called message
   */
  async invokeAI({
    messages,
    model,
    stopSequences,
    system,
    temperature,
    signal,
    timeout
  }) {
    const res = await this.runApi({
      body: JSON.stringify(formatBedrockBody({
        messages,
        stopSequences,
        system,
        temperature
      })),
      model,
      signal,
      timeout
    });
    return {
      message: res.completion.trim()
    };
  }
}
exports.BedrockConnector = BedrockConnector;
const formatBedrockBody = ({
  messages,
  stopSequences,
  temperature = 0,
  system
}) => ({
  anthropic_version: 'bedrock-2023-05-31',
  ...ensureMessageFormat(messages, system),
  max_tokens: _constants.DEFAULT_TOKEN_LIMIT,
  stop_sequences: stopSequences,
  temperature
});

/**
 * Ensures that the messages are in the correct format for the Bedrock API
 * If 2 user or 2 assistant messages are sent in a row, Bedrock throws an error
 * We combine the messages into a single message to avoid this error
 * @param messages
 */
const ensureMessageFormat = (messages, systemPrompt) => {
  let system = systemPrompt ? systemPrompt : '';
  const newMessages = messages.reduce((acc, m) => {
    const lastMessage = acc[acc.length - 1];
    if (lastMessage && lastMessage.role === m.role) {
      // Bedrock only accepts assistant and user roles.
      // If 2 user or 2 assistant messages are sent in a row, combine the messages into a single message
      return [...acc.slice(0, -1), {
        content: `${lastMessage.content}\n${m.content}`,
        role: m.role
      }];
    }
    if (m.role === 'system') {
      system = `${system.length ? `${system}\n` : ''}${m.content}`;
      return acc;
    }

    // force role outside of system to ensure it is either assistant or user
    return [...acc, {
      content: m.content,
      role: m.role === 'assistant' ? 'assistant' : 'user'
    }];
  }, []);
  return system.length ? {
    system,
    messages: newMessages
  } : {
    messages: newMessages
  };
};
function parseContent(content) {
  let parsedContent = '';
  if (content.length === 1 && content[0].type === 'text' && content[0].text) {
    parsedContent = content[0].text;
  } else if (content.length > 1) {
    parsedContent = content.reduce((acc, {
      text
    }) => text ? `${acc}\n${text}` : acc, '');
  }
  return parsedContent;
}
const usesDeprecatedArguments = body => {
  var _JSON$parse;
  return ((_JSON$parse = JSON.parse(body)) === null || _JSON$parse === void 0 ? void 0 : _JSON$parse.prompt) != null;
};