"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MetricCustomTransformGenerator = exports.INVALID_EQUATION_REGEX = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _ = require(".");
var _constants = require("../../../common/constants");
var _slo_transform_template = require("../../assets/transform_templates/slo_transform_template");
var _errors = require("../../errors");
var _aggregations = require("../aggregations");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INVALID_EQUATION_REGEX = exports.INVALID_EQUATION_REGEX = /[^A-Z|+|\-|\s|\d+|\.|\(|\)|\/|\*|>|<|=|\?|\:|&|\!|\|]+/g;
class MetricCustomTransformGenerator extends _.TransformGenerator {
  getTransformParams(slo) {
    if (!_sloSchema.metricCustomIndicatorSchema.is(slo.indicator)) {
      throw new _errors.InvalidTransformError(`Cannot handle SLO of indicator type: ${slo.indicator.type}`);
    }
    return (0, _slo_transform_template.getSLOTransformTemplate)(this.buildTransformId(slo), this.buildDescription(slo), this.buildSource(slo, slo.indicator), this.buildDestination(), this.buildCommonGroupBy(slo, slo.indicator.params.timestampField), this.buildAggregations(slo, slo.indicator), this.buildSettings(slo, slo.indicator.params.timestampField), slo);
  }
  buildTransformId(slo) {
    return (0, _constants.getSLOTransformId)(slo.id, slo.revision);
  }
  buildSource(slo, indicator) {
    return {
      index: (0, _.parseIndex)(indicator.params.index),
      runtime_mappings: this.buildCommonRuntimeMappings(slo),
      query: {
        bool: {
          filter: [{
            range: {
              [indicator.params.timestampField]: {
                gte: `now-${slo.timeWindow.duration.format()}/d`
              }
            }
          }, (0, _.getElasticsearchQueryOrThrow)(indicator.params.filter)]
        }
      }
    };
  }
  buildDestination() {
    return {
      pipeline: _constants.SLO_INGEST_PIPELINE_NAME,
      index: _constants.SLO_DESTINATION_INDEX_NAME
    };
  }
  buildAggregations(slo, indicator) {
    if (indicator.params.good.equation.match(INVALID_EQUATION_REGEX)) {
      throw new Error(`Invalid equation: ${indicator.params.good.equation}`);
    }
    if (indicator.params.total.equation.match(INVALID_EQUATION_REGEX)) {
      throw new Error(`Invalid equation: ${indicator.params.total.equation}`);
    }
    const getCustomMetricIndicatorAggregation = new _aggregations.GetCustomMetricIndicatorAggregation(indicator);
    return {
      ...getCustomMetricIndicatorAggregation.execute({
        type: 'good',
        aggregationKey: 'slo.numerator'
      }),
      ...getCustomMetricIndicatorAggregation.execute({
        type: 'total',
        aggregationKey: 'slo.denominator'
      }),
      ...(_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod) && {
        'slo.isGoodSlice': {
          bucket_script: {
            buckets_path: {
              goodEvents: 'slo.numerator>value',
              totalEvents: 'slo.denominator>value'
            },
            script: `params.goodEvents / params.totalEvents ${(0, _common.getTimesliceTargetComparator)(slo.objective.timesliceTarget)} ${slo.objective.timesliceTarget} ? 1 : 0`
          }
        }
      })
    };
  }
}
exports.MetricCustomTransformGenerator = MetricCustomTransformGenerator;