"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ApmTransactionDurationTransformGenerator = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _ = require(".");
var _constants = require("../../../common/constants");
var _slo_transform_template = require("../../assets/transform_templates/slo_transform_template");
var _errors = require("../../errors");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ApmTransactionDurationTransformGenerator extends _.TransformGenerator {
  getTransformParams(slo) {
    if (!_sloSchema.apmTransactionDurationIndicatorSchema.is(slo.indicator)) {
      throw new _errors.InvalidTransformError(`Cannot handle SLO of indicator type: ${slo.indicator.type}`);
    }
    return (0, _slo_transform_template.getSLOTransformTemplate)(this.buildTransformId(slo), this.buildDescription(slo), this.buildSource(slo, slo.indicator), this.buildDestination(), this.buildGroupBy(slo, slo.indicator), this.buildAggregations(slo, slo.indicator), this.buildSettings(slo), slo);
  }
  buildTransformId(slo) {
    return (0, _constants.getSLOTransformId)(slo.id, slo.revision);
  }
  buildGroupBy(slo, indicator) {
    // These groupBy fields must match the fields from the source query, otherwise
    // the transform will create permutations for each value present in the source.
    // E.g. if environment is not specified in the source query, but we include it in the groupBy,
    // we'll output documents for each environment value
    const extraGroupByFields = {
      ...(indicator.params.service !== _sloSchema.ALL_VALUE && {
        'service.name': {
          terms: {
            field: 'service.name'
          }
        }
      }),
      ...(indicator.params.environment !== _sloSchema.ALL_VALUE && {
        'service.environment': {
          terms: {
            field: 'service.environment'
          }
        }
      }),
      ...(indicator.params.transactionName !== _sloSchema.ALL_VALUE && {
        'transaction.name': {
          terms: {
            field: 'transaction.name'
          }
        }
      }),
      ...(indicator.params.transactionType !== _sloSchema.ALL_VALUE && {
        'transaction.type': {
          terms: {
            field: 'transaction.type'
          }
        }
      })
    };
    return this.buildCommonGroupBy(slo, '@timestamp', extraGroupByFields);
  }
  buildSource(slo, indicator) {
    const queryFilter = [{
      range: {
        '@timestamp': {
          gte: `now-${slo.timeWindow.duration.format()}/d`
        }
      }
    }];
    if (indicator.params.service !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'service.name': indicator.params.service
        }
      });
    }
    if (indicator.params.environment !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'service.environment': indicator.params.environment
        }
      });
    }
    if (indicator.params.transactionName !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'transaction.name': indicator.params.transactionName
        }
      });
    }
    if (indicator.params.transactionType !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'transaction.type': indicator.params.transactionType
        }
      });
    }
    if (!!indicator.params.filter) {
      queryFilter.push((0, _.getElasticsearchQueryOrThrow)(indicator.params.filter));
    }
    return {
      index: (0, _common.parseIndex)(indicator.params.index),
      runtime_mappings: this.buildCommonRuntimeMappings(slo),
      query: {
        bool: {
          filter: [{
            terms: {
              'processor.event': ['metric']
            }
          }, {
            term: {
              'metricset.name': 'transaction'
            }
          }, {
            exists: {
              field: 'transaction.duration.histogram'
            }
          }, ...queryFilter]
        }
      }
    };
  }
  buildDestination() {
    return {
      pipeline: _constants.SLO_INGEST_PIPELINE_NAME,
      index: _constants.SLO_DESTINATION_INDEX_NAME
    };
  }
  buildAggregations(slo, indicator) {
    // threshold is in ms (milliseconds), but apm data is stored in us (microseconds)
    const truncatedThreshold = Math.trunc(indicator.params.threshold * 1000);
    return {
      _numerator: {
        range: {
          field: 'transaction.duration.histogram',
          keyed: true,
          ranges: [{
            to: truncatedThreshold,
            key: 'target'
          }]
        }
      },
      'slo.numerator': {
        bucket_script: {
          buckets_path: {
            numerator: `_numerator['target']>_count`
          },
          script: 'params.numerator'
        }
      },
      'slo.denominator': {
        value_count: {
          field: 'transaction.duration.histogram'
        }
      },
      ...(_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod) && {
        'slo.isGoodSlice': {
          bucket_script: {
            buckets_path: {
              goodEvents: 'slo.numerator.value',
              totalEvents: 'slo.denominator.value'
            },
            script: `params.goodEvents / params.totalEvents ${(0, _common.getTimesliceTargetComparator)(slo.objective.timesliceTarget)} ${slo.objective.timesliceTarget} ? 1 : 0`
          }
        }
      })
    };
  }
}
exports.ApmTransactionDurationTransformGenerator = ApmTransactionDurationTransformGenerator;