"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GetPreviewData = void 0;
var _calculateAuto = require("@kbn/calculate-auto");
var _sloSchema = require("@kbn/slo-schema");
var _std = require("@kbn/std");
var _moment = _interopRequireDefault(require("moment"));
var _transform_generators = require("./transform_generators");
var _synthetics_availability = require("./transform_generators/synthetics_availability");
var _queries = require("../utils/queries");
var _services = require("../domain/services");
var _aggregations = require("./aggregations");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class GetPreviewData {
  constructor(esClient, spaceId) {
    this.esClient = esClient;
    this.spaceId = spaceId;
  }
  async getAPMTransactionDurationPreviewData(indicator, options) {
    var _result$aggregations$, _result$aggregations;
    const filter = [];
    this.getGroupingsFilter(options, filter);
    if (indicator.params.service !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'service.name': indicator.params.service
      }
    });
    if (indicator.params.environment !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'service.environment': indicator.params.environment
      }
    });
    if (indicator.params.transactionName !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'transaction.name': indicator.params.transactionName
      }
    });
    if (indicator.params.transactionType !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'transaction.type': indicator.params.transactionType
      }
    });
    if (!!indicator.params.filter) filter.push((0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter));
    const truncatedThreshold = Math.trunc(indicator.params.threshold * 1000);
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const result = await (0, _queries.typedSearch)(this.esClient, {
      index,
      size: 0,
      query: {
        bool: {
          filter: [{
            range: {
              '@timestamp': {
                gte: options.range.start,
                lte: options.range.end
              }
            }
          }, {
            terms: {
              'processor.event': ['metric']
            }
          }, {
            term: {
              'metricset.name': 'transaction'
            }
          }, {
            exists: {
              field: 'transaction.duration.histogram'
            }
          }, ...filter]
        }
      },
      aggs: {
        perMinute: {
          date_histogram: {
            field: '@timestamp',
            fixed_interval: options.interval,
            extended_bounds: {
              min: options.range.start,
              max: options.range.end
            }
          },
          aggs: {
            _good: {
              range: {
                field: 'transaction.duration.histogram',
                keyed: true,
                ranges: [{
                  to: truncatedThreshold,
                  key: 'target'
                }]
              }
            },
            good: {
              bucket_script: {
                buckets_path: {
                  _good: `_good['target']>_count`
                },
                script: 'params._good'
              }
            },
            total: {
              value_count: {
                field: 'transaction.duration.histogram'
              }
            }
          }
        }
      }
    });
    return (_result$aggregations$ = (_result$aggregations = result.aggregations) === null || _result$aggregations === void 0 ? void 0 : _result$aggregations.perMinute.buckets.map(bucket => {
      var _ref, _bucket$good, _bucket$total$value, _bucket$total;
      const good = (_ref = (_bucket$good = bucket.good) === null || _bucket$good === void 0 ? void 0 : _bucket$good.value) !== null && _ref !== void 0 ? _ref : 0;
      const total = (_bucket$total$value = (_bucket$total = bucket.total) === null || _bucket$total === void 0 ? void 0 : _bucket$total.value) !== null && _bucket$total$value !== void 0 ? _bucket$total$value : 0;
      return {
        date: bucket.key_as_string,
        sliValue: (0, _services.computeSLIForPreview)(good, total),
        events: {
          good,
          total,
          bad: total - good
        }
      };
    })) !== null && _result$aggregations$ !== void 0 ? _result$aggregations$ : [];
  }
  async getAPMTransactionErrorPreviewData(indicator, options) {
    var _result$aggregations2;
    const filter = [];
    this.getGroupingsFilter(options, filter);
    if (indicator.params.service !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'service.name': indicator.params.service
      }
    });
    if (indicator.params.environment !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'service.environment': indicator.params.environment
      }
    });
    if (indicator.params.transactionName !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'transaction.name': indicator.params.transactionName
      }
    });
    if (indicator.params.transactionType !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'transaction.type': indicator.params.transactionType
      }
    });
    if (!!indicator.params.filter) filter.push((0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter));
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const result = await this.esClient.search({
      index,
      size: 0,
      query: {
        bool: {
          filter: [{
            range: {
              '@timestamp': {
                gte: options.range.start,
                lte: options.range.end
              }
            }
          }, {
            term: {
              'metricset.name': 'transaction'
            }
          }, {
            terms: {
              'event.outcome': ['success', 'failure']
            }
          }, ...filter]
        }
      },
      aggs: {
        perMinute: {
          date_histogram: {
            field: '@timestamp',
            fixed_interval: options.interval,
            extended_bounds: {
              min: options.range.start,
              max: options.range.end
            }
          },
          aggs: {
            good: {
              filter: {
                bool: {
                  should: {
                    match: {
                      'event.outcome': 'success'
                    }
                  }
                }
              }
            },
            total: {
              filter: {
                match_all: {}
              }
            }
          }
        }
      }
    });

    // @ts-ignore buckets is not improperly typed
    return (_result$aggregations2 = result.aggregations) === null || _result$aggregations2 === void 0 ? void 0 : _result$aggregations2.perMinute.buckets.map(bucket => {
      var _bucket$good$doc_coun, _bucket$good2, _bucket$total$doc_cou, _bucket$total2, _bucket$good$doc_coun2, _bucket$good3, _bucket$total$doc_cou2, _bucket$total3;
      return {
        date: bucket.key_as_string,
        sliValue: !!bucket.good && !!bucket.total ? (0, _services.computeSLIForPreview)(bucket.good.doc_count, bucket.total.doc_count) : null,
        events: {
          good: (_bucket$good$doc_coun = (_bucket$good2 = bucket.good) === null || _bucket$good2 === void 0 ? void 0 : _bucket$good2.doc_count) !== null && _bucket$good$doc_coun !== void 0 ? _bucket$good$doc_coun : 0,
          bad: ((_bucket$total$doc_cou = (_bucket$total2 = bucket.total) === null || _bucket$total2 === void 0 ? void 0 : _bucket$total2.doc_count) !== null && _bucket$total$doc_cou !== void 0 ? _bucket$total$doc_cou : 0) - ((_bucket$good$doc_coun2 = (_bucket$good3 = bucket.good) === null || _bucket$good3 === void 0 ? void 0 : _bucket$good3.doc_count) !== null && _bucket$good$doc_coun2 !== void 0 ? _bucket$good$doc_coun2 : 0),
          total: (_bucket$total$doc_cou2 = (_bucket$total3 = bucket.total) === null || _bucket$total3 === void 0 ? void 0 : _bucket$total3.doc_count) !== null && _bucket$total$doc_cou2 !== void 0 ? _bucket$total$doc_cou2 : 0
        }
      };
    });
  }
  async getHistogramPreviewData(indicator, options) {
    var _result$aggregations3;
    const getHistogramIndicatorAggregations = new _aggregations.GetHistogramIndicatorAggregation(indicator);
    const filterQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter);
    const timestampField = indicator.params.timestampField;
    const filter = [{
      range: {
        [timestampField]: {
          gte: options.range.start,
          lte: options.range.end
        }
      }
    }, filterQuery];
    this.getGroupingsFilter(options, filter);
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const result = await this.esClient.search({
      index,
      size: 0,
      query: {
        bool: {
          filter
        }
      },
      aggs: {
        perMinute: {
          date_histogram: {
            field: timestampField,
            fixed_interval: options.interval,
            extended_bounds: {
              min: options.range.start,
              max: options.range.end
            }
          },
          aggs: {
            ...getHistogramIndicatorAggregations.execute({
              type: 'good',
              aggregationKey: 'good'
            }),
            ...getHistogramIndicatorAggregations.execute({
              type: 'total',
              aggregationKey: 'total'
            })
          }
        }
      }
    });

    // @ts-ignore buckets is not improperly typed
    return (_result$aggregations3 = result.aggregations) === null || _result$aggregations3 === void 0 ? void 0 : _result$aggregations3.perMinute.buckets.map(bucket => {
      var _bucket$good$value, _bucket$good4, _bucket$total$value2, _bucket$total4, _bucket$good$value2, _bucket$good5, _bucket$total$value3, _bucket$total5;
      return {
        date: bucket.key_as_string,
        sliValue: !!bucket.good && !!bucket.total ? (0, _services.computeSLIForPreview)(bucket.good.value, bucket.total.value) : null,
        events: {
          good: (_bucket$good$value = (_bucket$good4 = bucket.good) === null || _bucket$good4 === void 0 ? void 0 : _bucket$good4.value) !== null && _bucket$good$value !== void 0 ? _bucket$good$value : 0,
          bad: ((_bucket$total$value2 = (_bucket$total4 = bucket.total) === null || _bucket$total4 === void 0 ? void 0 : _bucket$total4.value) !== null && _bucket$total$value2 !== void 0 ? _bucket$total$value2 : 0) - ((_bucket$good$value2 = (_bucket$good5 = bucket.good) === null || _bucket$good5 === void 0 ? void 0 : _bucket$good5.value) !== null && _bucket$good$value2 !== void 0 ? _bucket$good$value2 : 0),
          total: (_bucket$total$value3 = (_bucket$total5 = bucket.total) === null || _bucket$total5 === void 0 ? void 0 : _bucket$total5.value) !== null && _bucket$total$value3 !== void 0 ? _bucket$total$value3 : 0
        }
      };
    });
  }
  async getCustomMetricPreviewData(indicator, options) {
    var _result$aggregations4;
    const timestampField = indicator.params.timestampField;
    const filterQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter);
    const getCustomMetricIndicatorAggregation = new _aggregations.GetCustomMetricIndicatorAggregation(indicator);
    const filter = [{
      range: {
        [timestampField]: {
          gte: options.range.start,
          lte: options.range.end
        }
      }
    }, filterQuery];
    this.getGroupingsFilter(options, filter);
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const result = await this.esClient.search({
      index,
      size: 0,
      query: {
        bool: {
          filter
        }
      },
      aggs: {
        perMinute: {
          date_histogram: {
            field: timestampField,
            fixed_interval: options.interval,
            extended_bounds: {
              min: options.range.start,
              max: options.range.end
            }
          },
          aggs: {
            ...getCustomMetricIndicatorAggregation.execute({
              type: 'good',
              aggregationKey: 'good'
            }),
            ...getCustomMetricIndicatorAggregation.execute({
              type: 'total',
              aggregationKey: 'total'
            })
          }
        }
      }
    });

    // @ts-ignore buckets is not improperly typed
    return (_result$aggregations4 = result.aggregations) === null || _result$aggregations4 === void 0 ? void 0 : _result$aggregations4.perMinute.buckets.map(bucket => {
      var _bucket$good$value3, _bucket$good6, _bucket$total$value4, _bucket$total6, _bucket$good$value4, _bucket$good7, _bucket$total$value5, _bucket$total7;
      return {
        date: bucket.key_as_string,
        sliValue: !!bucket.good && !!bucket.total ? (0, _services.computeSLIForPreview)(bucket.good.value, bucket.total.value) : null,
        events: {
          good: (_bucket$good$value3 = (_bucket$good6 = bucket.good) === null || _bucket$good6 === void 0 ? void 0 : _bucket$good6.value) !== null && _bucket$good$value3 !== void 0 ? _bucket$good$value3 : 0,
          bad: ((_bucket$total$value4 = (_bucket$total6 = bucket.total) === null || _bucket$total6 === void 0 ? void 0 : _bucket$total6.value) !== null && _bucket$total$value4 !== void 0 ? _bucket$total$value4 : 0) - ((_bucket$good$value4 = (_bucket$good7 = bucket.good) === null || _bucket$good7 === void 0 ? void 0 : _bucket$good7.value) !== null && _bucket$good$value4 !== void 0 ? _bucket$good$value4 : 0),
          total: (_bucket$total$value5 = (_bucket$total7 = bucket.total) === null || _bucket$total7 === void 0 ? void 0 : _bucket$total7.value) !== null && _bucket$total$value5 !== void 0 ? _bucket$total$value5 : 0
        }
      };
    });
  }
  async getTimesliceMetricPreviewData(indicator, options) {
    var _result$aggregations5;
    const timestampField = indicator.params.timestampField;
    const filterQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter);
    const getCustomMetricIndicatorAggregation = new _aggregations.GetTimesliceMetricIndicatorAggregation(indicator);
    const filter = [{
      range: {
        [timestampField]: {
          gte: options.range.start,
          lte: options.range.end
        }
      }
    }, filterQuery];
    this.getGroupingsFilter(options, filter);
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const result = await this.esClient.search({
      index,
      size: 0,
      query: {
        bool: {
          filter
        }
      },
      aggs: {
        perMinute: {
          date_histogram: {
            field: timestampField,
            fixed_interval: options.interval,
            extended_bounds: {
              min: options.range.start,
              max: options.range.end
            }
          },
          aggs: {
            ...getCustomMetricIndicatorAggregation.execute('metric')
          }
        }
      }
    });

    // @ts-ignore buckets is not improperly typed
    return (_result$aggregations5 = result.aggregations) === null || _result$aggregations5 === void 0 ? void 0 : _result$aggregations5.perMinute.buckets.map(bucket => ({
      date: bucket.key_as_string,
      sliValue: !!bucket.metric ? bucket.metric.value : null
    }));
  }
  async getCustomKQLPreviewData(indicator, options) {
    var _result$aggregations6;
    const filterQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter);
    const goodQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.good);
    const totalQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.total);
    const timestampField = indicator.params.timestampField;
    const filter = [{
      range: {
        [timestampField]: {
          gte: options.range.start,
          lte: options.range.end
        }
      }
    }, filterQuery];
    this.getGroupingsFilter(options, filter);
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const result = await this.esClient.search({
      index,
      size: 0,
      query: {
        bool: {
          filter
        }
      },
      aggs: {
        perMinute: {
          date_histogram: {
            field: timestampField,
            fixed_interval: options.interval,
            extended_bounds: {
              min: options.range.start,
              max: options.range.end
            }
          },
          aggs: {
            good: {
              filter: goodQuery
            },
            total: {
              filter: totalQuery
            }
          }
        }
      }
    });

    // @ts-ignore buckets is not improperly typed
    return (_result$aggregations6 = result.aggregations) === null || _result$aggregations6 === void 0 ? void 0 : _result$aggregations6.perMinute.buckets.map(bucket => {
      var _bucket$good$doc_coun3, _bucket$good8, _bucket$total$doc_cou3, _bucket$total8, _bucket$good$doc_coun4, _bucket$good9, _bucket$total$doc_cou4, _bucket$total9;
      return {
        date: bucket.key_as_string,
        sliValue: !!bucket.good && !!bucket.total ? (0, _services.computeSLIForPreview)(bucket.good.doc_count, bucket.total.doc_count) : null,
        events: {
          good: (_bucket$good$doc_coun3 = (_bucket$good8 = bucket.good) === null || _bucket$good8 === void 0 ? void 0 : _bucket$good8.doc_count) !== null && _bucket$good$doc_coun3 !== void 0 ? _bucket$good$doc_coun3 : 0,
          bad: ((_bucket$total$doc_cou3 = (_bucket$total8 = bucket.total) === null || _bucket$total8 === void 0 ? void 0 : _bucket$total8.doc_count) !== null && _bucket$total$doc_cou3 !== void 0 ? _bucket$total$doc_cou3 : 0) - ((_bucket$good$doc_coun4 = (_bucket$good9 = bucket.good) === null || _bucket$good9 === void 0 ? void 0 : _bucket$good9.doc_count) !== null && _bucket$good$doc_coun4 !== void 0 ? _bucket$good$doc_coun4 : 0),
          total: (_bucket$total$doc_cou4 = (_bucket$total9 = bucket.total) === null || _bucket$total9 === void 0 ? void 0 : _bucket$total9.doc_count) !== null && _bucket$total$doc_cou4 !== void 0 ? _bucket$total$doc_cou4 : 0
        }
      };
    });
  }
  getGroupingsFilter(options, filter) {
    const groupingsKeys = Object.keys(options.groupings || []);
    if (groupingsKeys.length) {
      groupingsKeys.forEach(key => {
        var _options$groupings;
        filter.push({
          term: {
            [key]: (_options$groupings = options.groupings) === null || _options$groupings === void 0 ? void 0 : _options$groupings[key]
          }
        });
      });
    } else if (options.instanceId !== _sloSchema.ALL_VALUE && options.groupBy) {
      filter.push({
        term: {
          [options.groupBy]: options.instanceId
        }
      });
    }
  }
  async getSyntheticsAvailabilityPreviewData(indicator, options) {
    var _result$aggregations7;
    const filter = [];
    const {
      monitorIds,
      tags,
      projects
    } = (0, _synthetics_availability.buildParamValues)({
      monitorIds: indicator.params.monitorIds || [],
      tags: indicator.params.tags || [],
      projects: indicator.params.projects || []
    });
    if (!monitorIds.includes(_sloSchema.ALL_VALUE) && monitorIds.length > 0) filter.push({
      terms: {
        'monitor.id': monitorIds
      }
    });
    if (!tags.includes(_sloSchema.ALL_VALUE) && tags.length > 0) filter.push({
      terms: {
        tags
      }
    });
    if (!projects.includes(_sloSchema.ALL_VALUE) && projects.length > 0) filter.push({
      terms: {
        'monitor.project.id': projects
      }
    });
    const index = options.remoteName ? `${options.remoteName}:${_constants.SYNTHETICS_INDEX_PATTERN}` : _constants.SYNTHETICS_INDEX_PATTERN;
    const result = await this.esClient.search({
      index,
      size: 0,
      query: {
        bool: {
          filter: [{
            range: {
              '@timestamp': {
                gte: options.range.start,
                lte: options.range.end
              }
            }
          }, {
            term: {
              'summary.final_attempt': true
            }
          }, {
            term: {
              'meta.space_id': this.spaceId
            }
          }, ...filter]
        }
      },
      aggs: {
        perMinute: {
          date_histogram: {
            field: '@timestamp',
            fixed_interval: '10m'
          },
          aggs: {
            good: {
              filter: {
                term: {
                  'monitor.status': 'up'
                }
              }
            },
            bad: {
              filter: {
                term: {
                  'monitor.status': 'down'
                }
              }
            },
            total: {
              filter: {
                match_all: {}
              }
            }
          }
        }
      }
    });
    const data = [];

    // @ts-ignore buckets is not improperly typed
    (_result$aggregations7 = result.aggregations) === null || _result$aggregations7 === void 0 ? void 0 : _result$aggregations7.perMinute.buckets.forEach(bucket => {
      var _bucket$good$doc_coun5, _bucket$good10, _bucket$bad$doc_count, _bucket$bad, _bucket$total$doc_cou5, _bucket$total10;
      const good = (_bucket$good$doc_coun5 = (_bucket$good10 = bucket.good) === null || _bucket$good10 === void 0 ? void 0 : _bucket$good10.doc_count) !== null && _bucket$good$doc_coun5 !== void 0 ? _bucket$good$doc_coun5 : 0;
      const bad = (_bucket$bad$doc_count = (_bucket$bad = bucket.bad) === null || _bucket$bad === void 0 ? void 0 : _bucket$bad.doc_count) !== null && _bucket$bad$doc_count !== void 0 ? _bucket$bad$doc_count : 0;
      const total = (_bucket$total$doc_cou5 = (_bucket$total10 = bucket.total) === null || _bucket$total10 === void 0 ? void 0 : _bucket$total10.doc_count) !== null && _bucket$total$doc_cou5 !== void 0 ? _bucket$total$doc_cou5 : 0;
      data.push({
        date: bucket.key_as_string,
        sliValue: (0, _services.computeSLIForPreview)(good, total),
        events: {
          good,
          bad,
          total
        }
      });
    });
    return data;
  }
  async execute(params) {
    try {
      var _params$objective, _calculateAuto$near$a, _calculateAuto$near;
      // If the time range is 24h or less, then we want to use a 1m bucket for the
      // Timeslice metric so that the chart is as close to the evaluation as possible.
      // Otherwise due to how the statistics work, the values might not look like
      // they've breached the threshold.
      const bucketSize = params.indicator.type === 'sli.metric.timeslice' && params.range.end - params.range.start <= 86_400_000 && (_params$objective = params.objective) !== null && _params$objective !== void 0 && _params$objective.timesliceWindow ? params.objective.timesliceWindow.asMinutes() : Math.max((_calculateAuto$near$a = (_calculateAuto$near = _calculateAuto.calculateAuto.near(100, _moment.default.duration(params.range.end - params.range.start, 'ms'))) === null || _calculateAuto$near === void 0 ? void 0 : _calculateAuto$near.asMinutes()) !== null && _calculateAuto$near$a !== void 0 ? _calculateAuto$near$a : 0, 1);
      const options = {
        instanceId: params.instanceId,
        range: params.range,
        groupBy: params.groupBy,
        remoteName: params.remoteName,
        groupings: params.groupings,
        interval: `${bucketSize}m`
      };
      const type = params.indicator.type;
      switch (type) {
        case 'sli.apm.transactionDuration':
          return this.getAPMTransactionDurationPreviewData(params.indicator, options);
        case 'sli.apm.transactionErrorRate':
          return this.getAPMTransactionErrorPreviewData(params.indicator, options);
        case 'sli.synthetics.availability':
          return this.getSyntheticsAvailabilityPreviewData(params.indicator, options);
        case 'sli.kql.custom':
          return this.getCustomKQLPreviewData(params.indicator, options);
        case 'sli.histogram.custom':
          return this.getHistogramPreviewData(params.indicator, options);
        case 'sli.metric.custom':
          return this.getCustomMetricPreviewData(params.indicator, options);
        case 'sli.metric.timeslice':
          return this.getTimesliceMetricPreviewData(params.indicator, options);
        default:
          (0, _std.assertNever)(type);
      }
    } catch (err) {
      return [];
    }
  }
}
exports.GetPreviewData = GetPreviewData;