"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRuleExecutor = void 0;
var _i18n = require("@kbn/i18n");
var _numeral = _interopRequireDefault(require("@elastic/numeral"));
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _server = require("@kbn/alerting-plugin/server");
var _lodash = require("lodash");
var _server2 = require("@kbn/spaces-plugin/server");
var _sloSchema = require("@kbn/slo-schema");
var _common = require("@kbn/observability-plugin/common");
var _slo = require("../../../../common/field_names/slo");
var _services = require("../../../services");
var _types = require("./types");
var _constants = require("../../../../common/constants");
var _evaluate = require("./lib/evaluate");
var _evaluate_dependencies = require("./lib/evaluate_dependencies");
var _should_suppress_instance_id = require("./lib/should_suppress_instance_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getRuleExecutor = ({
  basePath,
  alertsLocator
}) => async function executor(options) {
  var _alertsClient$getReco;
  const {
    services,
    params,
    logger,
    startedAt,
    spaceId,
    getTimeRange
  } = options;
  const {
    savedObjectsClient: soClient,
    scopedClusterClient: esClient,
    alertsClient
  } = services;
  if (!alertsClient) {
    throw new _server.AlertsClientError();
  }
  const sloRepository = new _services.KibanaSavedObjectsSLORepository(soClient, logger);
  const slo = await sloRepository.findById(params.sloId);
  if (!slo.enabled) {
    return {
      state: {}
    };
  }

  // We only need the end timestamp to base all of queries on. The length of the time range
  // doesn't matter for our use case since we allow the user to customize the window sizes,
  const {
    dateEnd
  } = getTimeRange('1m');
  const results = await (0, _evaluate.evaluate)(esClient.asCurrentUser, slo, params, new Date(dateEnd));
  const suppressResults = params.dependencies && results.some(res => res.shouldAlert) ? (await (0, _evaluate_dependencies.evaluateDependencies)(soClient, esClient.asCurrentUser, sloRepository, params.dependencies, new Date(dateEnd))).activeRules : [];
  if (results.length > 0) {
    const alertLimit = alertsClient.getAlertLimitValue();
    let hasReachedLimit = false;
    let scheduledActionsCount = 0;
    for (const result of results) {
      const {
        instanceId,
        shouldAlert,
        longWindowDuration,
        longWindowBurnRate,
        shortWindowDuration,
        shortWindowBurnRate,
        window: windowDef
      } = result;
      const urlQuery = instanceId === _sloSchema.ALL_VALUE ? '' : `?instanceId=${instanceId}`;
      const viewInAppUrl = (0, _server2.addSpaceIdToPath)(basePath.publicBaseUrl, spaceId, `/app/observability/slos/${slo.id}${urlQuery}`);
      if (shouldAlert) {
        const shouldSuppress = (0, _should_suppress_instance_id.shouldSuppressInstanceId)(suppressResults, instanceId);
        if (scheduledActionsCount >= alertLimit) {
          // need to set this so that warning is displayed in the UI and in the logs
          hasReachedLimit = true;
          break; // once limit is reached, we break out of the loop and don't schedule any more alerts
        }
        const reason = buildReason(instanceId, windowDef.actionGroup, longWindowDuration, longWindowBurnRate, shortWindowDuration, shortWindowBurnRate, windowDef, shouldSuppress);
        const alertId = instanceId;
        const actionGroup = shouldSuppress ? _constants.SUPPRESSED_PRIORITY_ACTION.id : windowDef.actionGroup;
        const {
          uuid,
          start
        } = alertsClient.report({
          id: alertId,
          actionGroup,
          state: {
            alertState: _types.AlertStates.ALERT
          },
          payload: {
            [_ruleDataUtils.ALERT_REASON]: reason,
            [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: windowDef.burnRateThreshold,
            [_ruleDataUtils.ALERT_EVALUATION_VALUE]: Math.min(longWindowBurnRate, shortWindowBurnRate),
            [_slo.SLO_ID_FIELD]: slo.id,
            [_slo.SLO_REVISION_FIELD]: slo.revision,
            [_slo.SLO_INSTANCE_ID_FIELD]: instanceId
          }
        });
        const indexedStartedAt = start !== null && start !== void 0 ? start : startedAt.toISOString();
        const alertDetailsUrl = await (0, _common.getAlertUrl)(uuid, spaceId, indexedStartedAt, alertsLocator, basePath.publicBaseUrl);
        const context = {
          alertDetailsUrl,
          reason,
          longWindow: {
            burnRate: longWindowBurnRate,
            duration: longWindowDuration.format()
          },
          shortWindow: {
            burnRate: shortWindowBurnRate,
            duration: shortWindowDuration.format()
          },
          burnRateThreshold: windowDef.burnRateThreshold,
          timestamp: startedAt.toISOString(),
          viewInAppUrl,
          sloId: slo.id,
          sloName: slo.name,
          sloInstanceId: instanceId,
          slo,
          suppressedAction: shouldSuppress ? windowDef.actionGroup : null
        };
        alertsClient.setAlertData({
          id: alertId,
          context
        });
        scheduledActionsCount++;
      }
    }
    alertsClient.setAlertLimitReached(hasReachedLimit);
  }
  const recoveredAlerts = (_alertsClient$getReco = alertsClient.getRecoveredAlerts()) !== null && _alertsClient$getReco !== void 0 ? _alertsClient$getReco : [];
  for (const recoveredAlert of recoveredAlerts) {
    var _recoveredAlert$alert;
    const alertId = recoveredAlert.alert.getId();
    const indexedStartedAt = (_recoveredAlert$alert = recoveredAlert.alert.getStart()) !== null && _recoveredAlert$alert !== void 0 ? _recoveredAlert$alert : startedAt.toISOString();
    const alertUuid = recoveredAlert.alert.getUuid();
    const alertDetailsUrl = await (0, _common.getAlertUrl)(alertUuid, spaceId, indexedStartedAt, alertsLocator, basePath.publicBaseUrl);
    const urlQuery = alertId === _sloSchema.ALL_VALUE ? '' : `?instanceId=${alertId}`;
    const viewInAppUrl = (0, _server2.addSpaceIdToPath)(basePath.publicBaseUrl, spaceId, `/app/observability/slos/${slo.id}${urlQuery}`);
    const context = {
      timestamp: startedAt.toISOString(),
      viewInAppUrl,
      alertDetailsUrl,
      sloId: slo.id,
      sloName: slo.name,
      sloInstanceId: alertId
    };
    alertsClient.setAlertData({
      id: alertId,
      context
    });
  }
  return {
    state: {}
  };
};
exports.getRuleExecutor = getRuleExecutor;
function getActionGroupName(id) {
  switch (id) {
    case _constants.HIGH_PRIORITY_ACTION.id:
      return _constants.HIGH_PRIORITY_ACTION.name;
    case _constants.MEDIUM_PRIORITY_ACTION.id:
      return _constants.MEDIUM_PRIORITY_ACTION.name;
    case _constants.LOW_PRIORITY_ACTION.id:
      return _constants.LOW_PRIORITY_ACTION.name;
    default:
      return _constants.ALERT_ACTION.name;
  }
}
function buildReason(instanceId, actionGroup, longWindowDuration, longWindowBurnRate, shortWindowDuration, shortWindowBurnRate, windowDef, suppressed) {
  const actionGroupName = suppressed ? `${(0, _lodash.upperCase)(_constants.SUPPRESSED_PRIORITY_ACTION.name)} - ${(0, _lodash.upperCase)(getActionGroupName(actionGroup))}` : (0, _lodash.upperCase)(getActionGroupName(actionGroup));
  if (instanceId === _sloSchema.ALL_VALUE) {
    return _i18n.i18n.translate('xpack.slo.alerting.burnRate.reason', {
      defaultMessage: '{actionGroupName}: The burn rate for the past {longWindowDuration} is {longWindowBurnRate} and for the past {shortWindowDuration} is {shortWindowBurnRate}. Alert when above {burnRateThreshold} for both windows',
      values: {
        actionGroupName,
        longWindowDuration: longWindowDuration.format(),
        longWindowBurnRate: (0, _numeral.default)(longWindowBurnRate).format('0.[00]'),
        shortWindowDuration: shortWindowDuration.format(),
        shortWindowBurnRate: (0, _numeral.default)(shortWindowBurnRate).format('0.[00]'),
        burnRateThreshold: windowDef.burnRateThreshold
      }
    });
  }
  return _i18n.i18n.translate('xpack.slo.alerting.burnRate.reasonForInstanceId', {
    defaultMessage: '{actionGroupName}: The burn rate for the past {longWindowDuration} is {longWindowBurnRate} and for the past {shortWindowDuration} is {shortWindowBurnRate} for {instanceId}. Alert when above {burnRateThreshold} for both windows',
    values: {
      actionGroupName,
      longWindowDuration: longWindowDuration.format(),
      longWindowBurnRate: (0, _numeral.default)(longWindowBurnRate).format('0.[00]'),
      shortWindowDuration: shortWindowDuration.format(),
      shortWindowBurnRate: (0, _numeral.default)(shortWindowBurnRate).format('0.[00]'),
      burnRateThreshold: windowDef.burnRateThreshold,
      instanceId
    }
  });
}