"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerIndicesRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _lib = require("@kbn/search-index-documents/lib");
var _types = require("@kbn/search-index-documents/types");
var _fetch_indices = require("../lib/indices/fetch_indices");
var _fetch_index = require("../lib/indices/fetch_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerIndicesRoutes = ({
  router,
  security
}) => {
  router.get({
    path: '/internal/serverless_search/indices',
    validate: {
      query: _configSchema.schema.object({
        from: _configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }),
        search_query: _configSchema.schema.maybe(_configSchema.schema.string()),
        size: _configSchema.schema.number({
          defaultValue: 20,
          min: 0
        })
      })
    }
  }, async (context, request, response) => {
    const client = (await context.core).elasticsearch.client.asCurrentUser;
    const user = security.authc.getCurrentUser(request);
    if (!user) {
      return response.customError({
        statusCode: 502,
        body: 'Could not retrieve current user, security plugin is not ready'
      });
    }
    const {
      from,
      size,
      search_query: searchQuery
    } = request.query;
    const indices = await (0, _fetch_indices.fetchIndices)(client, from, size, searchQuery);
    return response.ok({
      body: {
        indices
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  });
  router.get({
    path: '/internal/serverless_search/index_names',
    validate: {
      query: _configSchema.schema.object({
        query: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const client = (await context.core).elasticsearch.client.asCurrentUser;
    const result = await client.indices.get({
      expand_wildcards: 'open',
      index: request.query.query ? `*${request.query.query}*` : '*'
    });
    return response.ok({
      body: {
        index_names: Object.keys(result || {}).filter(indexName => !isHidden(result[indexName]) && !isClosed(result[indexName]))
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  });
  router.get({
    path: '/internal/serverless_search/index/{indexName}',
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const body = await (0, _fetch_index.fetchIndex)(client.asCurrentUser, request.params.indexName);
    return body ? response.ok({
      body,
      headers: {
        'content-type': 'application/json'
      }
    }) : response.notFound();
  });
  router.post({
    path: '/internal/serverless_search/indices/{index_name}/search',
    validate: {
      body: _configSchema.schema.object({
        searchQuery: _configSchema.schema.string({
          defaultValue: ''
        })
      }),
      params: _configSchema.schema.object({
        index_name: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        page: _configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }),
        size: _configSchema.schema.number({
          defaultValue: _types.DEFAULT_DOCS_PER_PAGE,
          min: 0
        })
      })
    }
  }, async (context, request, response) => {
    const client = (await context.core).elasticsearch.client.asCurrentUser;
    const indexName = decodeURIComponent(request.params.index_name);
    const searchQuery = request.body.searchQuery;
    const {
      page = 0,
      size = _types.DEFAULT_DOCS_PER_PAGE
    } = request.query;
    const from = page * size;
    const searchResults = await (0, _lib.fetchSearchResults)(client, indexName, searchQuery, from, size);
    return response.ok({
      body: {
        results: searchResults
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  });
};
exports.registerIndicesRoutes = registerIndicesRoutes;
function isHidden(index) {
  var _index$settings, _index$settings$index, _index$settings2, _index$settings2$inde;
  return (index === null || index === void 0 ? void 0 : (_index$settings = index.settings) === null || _index$settings === void 0 ? void 0 : (_index$settings$index = _index$settings.index) === null || _index$settings$index === void 0 ? void 0 : _index$settings$index.hidden) === true || (index === null || index === void 0 ? void 0 : (_index$settings2 = index.settings) === null || _index$settings2 === void 0 ? void 0 : (_index$settings2$inde = _index$settings2.index) === null || _index$settings2$inde === void 0 ? void 0 : _index$settings2$inde.hidden) === 'true';
}
function isClosed(index) {
  var _index$settings3, _index$settings3$inde, _index$settings4, _index$settings4$inde;
  return (index === null || index === void 0 ? void 0 : (_index$settings3 = index.settings) === null || _index$settings3 === void 0 ? void 0 : (_index$settings3$inde = _index$settings3.index) === null || _index$settings3$inde === void 0 ? void 0 : _index$settings3$inde.verified_before_close) === true || (index === null || index === void 0 ? void 0 : (_index$settings4 = index.settings) === null || _index$settings4 === void 0 ? void 0 : (_index$settings4$inde = _index$settings4.index) === null || _index$settings4$inde === void 0 ? void 0 : _index$settings4$inde.verified_before_close) === 'true';
}