"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VERSION = exports.SecurityUsageReportingTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/task-manager-plugin/server");
var _services = require("../common/services");
var _task_state = require("./task_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SCOPE = ['serverlessSecurity'];
const TIMEOUT = '1m';
const VERSION = exports.VERSION = '1.0.0';
class SecurityUsageReportingTask {
  constructor(setupContract) {
    (0, _defineProperty2.default)(this, "wasStarted", false);
    (0, _defineProperty2.default)(this, "cloudSetup", void 0);
    (0, _defineProperty2.default)(this, "taskType", void 0);
    (0, _defineProperty2.default)(this, "version", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "start", async ({
      taskManager,
      interval
    }) => {
      if (!taskManager) {
        this.logger.error(`missing required taskmanager service during start of ${this.taskType}`);
        return;
      }
      this.wasStarted = true;
      try {
        await taskManager.ensureScheduled({
          id: this.taskId,
          taskType: this.taskType,
          scope: SCOPE,
          schedule: {
            interval
          },
          state: _task_state.emptyState,
          params: {
            version: this.version
          }
        });
      } catch (e) {
        this.logger.error(`Error scheduling task ${this.taskType}, received ${e.message}`);
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async (taskInstance, core, meteringCallback) => {
      var _usageReportResponse;
      // if task was not `.start()`'d yet, then exit
      if (!this.wasStarted) {
        this.logger.debug('[runTask()] Aborted. Task not started yet');
        return {
          state: taskInstance.state
        };
      }
      // Check that this task is current
      if (taskInstance.id !== this.taskId) {
        // old task, die
        (0, _server.throwUnrecoverableError)(new Error('Outdated task version'));
      }
      const [{
        elasticsearch
      }] = await core.getStartServices();
      const esClient = elasticsearch.client.asInternalUser;
      const epochDate = new Date();
      epochDate.setFullYear(1969);
      const lastSuccessfulReport = taskInstance.state.lastSuccessfulReport && new Date(taskInstance.state.lastSuccessfulReport) || epochDate;
      let usageRecords = [];
      let latestRecordTimestamp;
      let shouldRunAgain = false;
      // save usage record query time so we can use it to know where
      // the next query range should start
      const meteringCallbackTime = new Date();
      try {
        var _meteringCallbackResp, _meteringCallbackResp2;
        const meteringCallbackResponse = await meteringCallback({
          esClient,
          cloudSetup: this.cloudSetup,
          logger: this.logger,
          taskId: this.taskId,
          lastSuccessfulReport,
          abortController: this.abortController,
          config: this.config
        });
        usageRecords = (_meteringCallbackResp = meteringCallbackResponse.records) !== null && _meteringCallbackResp !== void 0 ? _meteringCallbackResp : [];
        latestRecordTimestamp = meteringCallbackResponse.latestTimestamp;
        shouldRunAgain = (_meteringCallbackResp2 = meteringCallbackResponse.shouldRunAgain) !== null && _meteringCallbackResp2 !== void 0 ? _meteringCallbackResp2 : false;
      } catch (err) {
        this.logger.error(`failed to retrieve usage records starting from ${lastSuccessfulReport.toISOString()}: ${err}`);
        return {
          state: taskInstance.state,
          runAt: new Date()
        };
      }
      this.logger.debug(`received usage records: ${JSON.stringify(usageRecords)}`);
      let usageReportResponse;
      if (usageRecords.length !== 0) {
        try {
          usageReportResponse = await _services.usageReportingService.reportUsage(usageRecords);
          if (!usageReportResponse.ok) {
            const errorResponse = await usageReportResponse.json();
            this.logger.warn(`API error ${usageReportResponse.status}, ${errorResponse}`);
            return {
              state: taskInstance.state,
              runAt: new Date()
            };
          }
          this.logger.info(`(${usageRecords.length}) usage records starting from ${lastSuccessfulReport.toISOString()} were sent successfully: ${usageReportResponse.status}, ${usageReportResponse.statusText}`);
        } catch (err) {
          this.logger.warn(`Failed to send (${usageRecords.length}) usage records starting from ${lastSuccessfulReport.toISOString()}: ${err} `);
          shouldRunAgain = true;
        }
      }
      const state = {
        lastSuccessfulReport: !usageRecords.length || ((_usageReportResponse = usageReportResponse) === null || _usageReportResponse === void 0 ? void 0 : _usageReportResponse.status) === 201 ? (latestRecordTimestamp || meteringCallbackTime).toISOString() : lastSuccessfulReport.toISOString()
      };
      return shouldRunAgain ? {
        state,
        runAt: new Date()
      } : {
        state
      };
    });
    const {
      core: _core,
      logFactory,
      config,
      taskManager: _taskManager,
      cloudSetup,
      taskType,
      taskTitle,
      version,
      meteringCallback: _meteringCallback
    } = setupContract;
    this.cloudSetup = cloudSetup;
    this.taskType = taskType;
    this.version = version;
    this.logger = logFactory.get(this.taskId);
    this.config = config;
    try {
      _taskManager.registerTaskDefinitions({
        [taskType]: {
          title: taskTitle,
          timeout: TIMEOUT,
          stateSchemaByVersion: _task_state.stateSchemaByVersion,
          createTaskRunner: ({
            taskInstance
          }) => {
            return {
              run: async () => {
                return this.runTask(taskInstance, _core, _meteringCallback);
              },
              cancel: async () => {}
            };
          }
        }
      });
      this.logger.info(`Scheduled task successfully ${taskTitle}`);
    } catch (err) {
      this.logger.error(`Failed to setup task ${taskType}, ${err} `);
    }
  }
  get taskId() {
    return `${this.taskType}:${this.version}`;
  }
}
exports.SecurityUsageReportingTask = SecurityUsageReportingTask;