"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.pickSavedNote = exports.persistNote = exports.getNotesByTimelineId = exports.getNote = exports.deleteNotesByTimelineId = exports.deleteNote = exports.convertSavedObjectToSavedNote = void 0;
var _PathReporter = require("io-ts/lib/PathReporter");
var _fp = require("lodash/fp");
var _uuid = require("uuid");
var _pipeable = require("fp-ts/lib/pipeable");
var _Either = require("fp-ts/lib/Either");
var _function = require("fp-ts/lib/function");
var _userProfileComponents = require("@kbn/user-profile-components");
var _constants = require("../../../../../common/constants");
var _saved_object = require("../../../../../common/types/timeline/note/saved_object");
var _notes = require("../../saved_object_mappings/notes");
var _saved_object_mappings = require("../../saved_object_mappings");
var _field_migrator = require("./field_migrator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deleteNotesByTimelineId = async (request, timelineId) => {
  const options = {
    type: _notes.noteSavedObjectType,
    hasReference: {
      type: _saved_object_mappings.timelineSavedObjectType,
      id: timelineId
    }
  };
  const notesToBeDeleted = await getAllSavedNote(request, options);
  const savedObjectsClient = (await request.context.core).savedObjects.client;
  const noteObjects = notesToBeDeleted.notes.map(note => {
    return {
      id: note.noteId,
      type: _notes.noteSavedObjectType
    };
  });
  await savedObjectsClient.bulkDelete(noteObjects);
};
exports.deleteNotesByTimelineId = deleteNotesByTimelineId;
const deleteNote = async ({
  request,
  noteId
}) => {
  const savedObjectsClient = (await request.context.core).savedObjects.client;
  await savedObjectsClient.delete(_notes.noteSavedObjectType, noteId);
};
exports.deleteNote = deleteNote;
const getNote = async (request, noteId) => {
  return getSavedNote(request, noteId);
};
exports.getNote = getNote;
const getNotesByTimelineId = async (request, timelineId) => {
  const options = {
    type: _notes.noteSavedObjectType,
    hasReference: {
      type: _saved_object_mappings.timelineSavedObjectType,
      id: timelineId
    }
  };
  const notesByTimelineId = await getAllSavedNote(request, options);
  return notesByTimelineId.notes;
};
exports.getNotesByTimelineId = getNotesByTimelineId;
const persistNote = async ({
  request,
  noteId,
  note,
  overrideOwner = true
}) => {
  try {
    if (noteId == null) {
      return await createNote({
        request,
        noteId,
        note,
        overrideOwner
      });
    }

    // Update existing note
    return await updateNote({
      request,
      noteId,
      note,
      overrideOwner
    });
  } catch (err) {
    if ((0, _fp.getOr)(null, 'output.statusCode', err) === 403) {
      const noteToReturn = {
        ...note,
        noteId: (0, _uuid.v1)(),
        version: '',
        timelineId: ''
      };
      return {
        code: 403,
        message: err.message,
        note: noteToReturn
      };
    }
    throw err;
  }
};
exports.persistNote = persistNote;
const createNote = async ({
  request,
  noteId,
  note,
  overrideOwner = true
}) => {
  const savedObjectsClient = (await request.context.core).savedObjects.client;
  const userInfo = request.user;
  const noteWithCreator = overrideOwner ? pickSavedNote(noteId, {
    ...note
  }, userInfo) : note;
  const {
    transformedFields: migratedAttributes,
    references
  } = _field_migrator.noteFieldsMigrator.extractFieldsToReferences({
    data: noteWithCreator
  });
  const noteAttributes = {
    eventId: migratedAttributes.eventId,
    note: migratedAttributes.note,
    created: migratedAttributes.created,
    createdBy: migratedAttributes.createdBy,
    updated: migratedAttributes.updated,
    updatedBy: migratedAttributes.updatedBy
  };
  const createdNote = await savedObjectsClient.create(_notes.noteSavedObjectType, noteAttributes, {
    references
  });
  const repopulatedSavedObject = _field_migrator.noteFieldsMigrator.populateFieldsFromReferences(createdNote);
  const convertedNote = convertSavedObjectToSavedNote(repopulatedSavedObject);

  // Create new note
  return {
    code: 200,
    message: 'success',
    note: convertedNote
  };
};
const updateNote = async ({
  request,
  noteId,
  note,
  overrideOwner = true
}) => {
  const savedObjectsClient = (await request.context.core).savedObjects.client;
  const userInfo = request.user;
  const existingNote = await savedObjectsClient.get(_notes.noteSavedObjectType, noteId);
  const noteWithCreator = overrideOwner ? pickSavedNote(noteId, note, userInfo) : note;
  const {
    transformedFields: migratedPatchAttributes,
    references
  } = _field_migrator.noteFieldsMigrator.extractFieldsToReferences({
    data: noteWithCreator,
    existingReferences: existingNote.references
  });
  const noteAttributes = {
    eventId: migratedPatchAttributes.eventId,
    note: migratedPatchAttributes.note,
    created: migratedPatchAttributes.created,
    createdBy: migratedPatchAttributes.createdBy,
    updated: migratedPatchAttributes.updated,
    updatedBy: migratedPatchAttributes.updatedBy
  };
  const updatedNote = await savedObjectsClient.update(_notes.noteSavedObjectType, noteId, noteAttributes, {
    version: existingNote.version || undefined,
    references
  });
  const populatedNote = _field_migrator.noteFieldsMigrator.populateFieldsFromReferencesForPatch({
    dataBeforeRequest: note,
    dataReturnedFromRequest: updatedNote
  });
  const convertedNote = convertSavedObjectToSavedNote(populatedNote);
  return {
    code: 200,
    message: 'success',
    note: convertedNote
  };
};
const getSavedNote = async (request, NoteId) => {
  const savedObjectsClient = (await request.context.core).savedObjects.client;
  const savedObject = await savedObjectsClient.get(_notes.noteSavedObjectType, NoteId);
  const populatedNote = _field_migrator.noteFieldsMigrator.populateFieldsFromReferences(savedObject);
  return convertSavedObjectToSavedNote(populatedNote);
};
const getAllSavedNote = async (request, options) => {
  const savedObjectsClient = (await request.context.core).savedObjects.client;
  const savedObjects = await savedObjectsClient.find(options);
  return {
    totalCount: savedObjects.total,
    notes: savedObjects.saved_objects.map(savedObject => {
      const populatedNote = _field_migrator.noteFieldsMigrator.populateFieldsFromReferences(savedObject);
      return convertSavedObjectToSavedNote(populatedNote);
    })
  };
};
const convertSavedObjectToSavedNote = savedObject => (0, _pipeable.pipe)(_saved_object.SavedObjectNoteRuntimeType.decode(savedObject), (0, _Either.map)(savedNote => {
  return {
    noteId: savedNote.id,
    version: savedNote.version,
    timelineId: savedNote.attributes.timelineId,
    eventId: savedNote.attributes.eventId,
    note: savedNote.attributes.note,
    created: savedNote.attributes.created,
    createdBy: savedNote.attributes.createdBy,
    updated: savedNote.attributes.updated,
    updatedBy: savedNote.attributes.updatedBy
  };
}), (0, _Either.fold)(errors => {
  throw new Error((0, _PathReporter.failure)(errors).join('\n'));
}, _function.identity));
exports.convertSavedObjectToSavedNote = convertSavedObjectToSavedNote;
const pickSavedNote = (noteId, savedNote, userInfo) => {
  if (noteId == null) {
    savedNote.created = new Date().valueOf();
    savedNote.createdBy = userInfo ? (0, _userProfileComponents.getUserDisplayName)(userInfo) : _constants.UNAUTHENTICATED_USER;
  }
  savedNote.updated = new Date().valueOf();
  savedNote.updatedBy = userInfo ? (0, _userProfileComponents.getUserDisplayName)(userInfo) : _constants.UNAUTHENTICATED_USER;
  return savedNote;
};
exports.pickSavedNote = pickSavedNote;