"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.artifactService = exports.Artifact = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _axios = _interopRequireDefault(require("axios"));
var _lodash = require("lodash");
var _admZip = _interopRequireDefault(require("adm-zip"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class Artifact {
  constructor() {
    (0, _defineProperty2.default)(this, "manifestUrl", void 0);
    (0, _defineProperty2.default)(this, "CDN_URL", 'https://artifacts.security.elastic.co');
    (0, _defineProperty2.default)(this, "AXIOS_TIMEOUT_MS", 10_000);
    (0, _defineProperty2.default)(this, "receiver", void 0);
    (0, _defineProperty2.default)(this, "esClusterInfo", void 0);
    (0, _defineProperty2.default)(this, "cache", new Map());
  }
  async start(receiver) {
    var _this$esClusterInfo, _this$esClusterInfo$v;
    this.receiver = receiver;
    this.esClusterInfo = await this.receiver.fetchClusterInfo();
    if ((_this$esClusterInfo = this.esClusterInfo) !== null && _this$esClusterInfo !== void 0 && (_this$esClusterInfo$v = _this$esClusterInfo.version) !== null && _this$esClusterInfo$v !== void 0 && _this$esClusterInfo$v.number) {
      const version = this.esClusterInfo.version.number.substring(0, this.esClusterInfo.version.number.indexOf('-')) || this.esClusterInfo.version.number;
      this.manifestUrl = `${this.CDN_URL}/downloads/kibana/manifest/artifacts-${version}.zip`;
    }
  }
  async getArtifact(name) {
    return _axios.default.get(this.getManifestUrl(), {
      headers: this.headers(name),
      timeout: this.AXIOS_TIMEOUT_MS,
      validateStatus: status => status < 500,
      responseType: 'arraybuffer'
    }).then(async response => {
      switch (response.status) {
        case 200:
          const manifest = {
            data: await this.getManifest(name, response.data),
            notModified: false
          };
          // only update etag if we got a valid manifest
          if (response.headers && response.headers.etag) {
            var _response$headers$eta, _response$headers;
            const cacheEntry = {
              manifest: {
                ...manifest,
                notModified: true
              },
              etag: (_response$headers$eta = (_response$headers = response.headers) === null || _response$headers === void 0 ? void 0 : _response$headers.etag) !== null && _response$headers$eta !== void 0 ? _response$headers$eta : ''
            };
            this.cache.set(name, cacheEntry);
          }
          return (0, _lodash.cloneDeep)(manifest);
        case 304:
          return (0, _lodash.cloneDeep)(this.getCachedManifest(name));
        case 404:
          throw Error(`No manifest resource found at url: ${this.manifestUrl}`);
        default:
          throw Error(`Failed to download manifest, unexpected status code: ${response.status}`);
      }
    });
  }
  getManifestUrl() {
    if (this.manifestUrl) {
      return this.manifestUrl;
    } else {
      var _this$esClusterInfo2, _this$esClusterInfo2$;
      throw Error(`No manifest url for version ${(_this$esClusterInfo2 = this.esClusterInfo) === null || _this$esClusterInfo2 === void 0 ? void 0 : (_this$esClusterInfo2$ = _this$esClusterInfo2.version) === null || _this$esClusterInfo2$ === void 0 ? void 0 : _this$esClusterInfo2$.number}`);
    }
  }
  getCachedManifest(name) {
    const entry = this.cache.get(name);
    if (!entry) {
      throw Error(`No cached manifest for name ${name}`);
    }
    return entry.manifest;
  }
  async getManifest(name, data) {
    var _manifest$artifacts$n;
    const zip = new _admZip.default(data);
    const manifestFile = zip.getEntries().find(entry => {
      return entry.entryName === 'manifest.json';
    });
    if (!manifestFile) {
      throw Error('No manifest.json in artifact zip');
    }
    const manifest = JSON.parse(manifestFile.getData().toString());
    const relativeUrl = (_manifest$artifacts$n = manifest.artifacts[name]) === null || _manifest$artifacts$n === void 0 ? void 0 : _manifest$artifacts$n.relative_url;
    if (relativeUrl) {
      const url = `${this.CDN_URL}${relativeUrl}`;
      const artifactResponse = await _axios.default.get(url, {
        timeout: this.AXIOS_TIMEOUT_MS
      });
      return artifactResponse.data;
    } else {
      throw Error(`No artifact for name ${name}`);
    }
  }

  // morre info https://www.rfc-editor.org/rfc/rfc9110#name-etag
  headers(name) {
    var _this$cache$get;
    const etag = (_this$cache$get = this.cache.get(name)) === null || _this$cache$get === void 0 ? void 0 : _this$cache$get.etag;
    if (etag) {
      return {
        'If-None-Match': etag
      };
    }
    return {};
  }
}
exports.Artifact = Artifact;
const artifactService = exports.artifactService = new Artifact();