"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AssetCriticalityDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("@kbn/alerting-plugin/common");
var _create_or_update_index = require("../utils/create_or_update_index");
var _asset_criticality = require("../../../../common/entity_analytics/asset_criticality");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CRITICALITY_RESPONSE_SIZE = 100_000;
const DEFAULT_CRITICALITY_RESPONSE_SIZE = 1_000;
const createId = ({
  idField,
  idValue
}) => `${idField}:${idValue}`;
class AssetCriticalityDataClient {
  constructor(options) {
    /**
     * Bulk upsert asset criticality records from a stream.
     * @param recordsStream a stream of records to upsert, records may also be an error e.g if there was an error parsing
     * @param flushBytes how big elasticsearch bulk requests should be before they are sent
     * @param retries the number of times to retry a failed bulk request
     * @returns an object containing the number of records updated, created, errored, and the total number of records processed
     * @throws an error if the stream emits an error
     * @remarks
     * - The stream must emit records in the format of {@link AssetCriticalityUpsert} or an error instance
     * - The stream must emit records in the order they should be upserted
     * - The stream must emit records in a valid JSON format
     * - We allow errors to be emitted in the stream to allow for partial upserts and to maintain the order of records
     **/
    (0, _defineProperty2.default)(this, "bulkUpsertFromStream", async ({
      recordsStream,
      flushBytes,
      retries
    }) => {
      const errors = [];
      const stats = {
        successful: 0,
        failed: 0,
        total: 0
      };
      let streamIndex = 0;
      const recordGenerator = async function* () {
        for await (const untypedRecord of recordsStream) {
          const record = untypedRecord;
          stats.total++;
          if (record instanceof Error) {
            stats.failed++;
            errors.push({
              message: record.message,
              index: streamIndex
            });
          } else {
            yield {
              record,
              index: streamIndex
            };
          }
          streamIndex++;
        }
      };
      const {
        failed,
        successful
      } = await this.options.esClient.helpers.bulk({
        datasource: recordGenerator(),
        index: this.getIndex(),
        flushBytes,
        retries,
        refreshOnCompletion: true,
        // refresh the index after all records are processed
        onDocument: ({
          record
        }) => [{
          update: {
            _id: createId(record)
          }
        }, {
          doc: {
            id_field: record.idField,
            id_value: record.idValue,
            criticality_level: record.criticalityLevel,
            '@timestamp': new Date().toISOString()
          },
          doc_as_upsert: true
        }],
        onDrop: ({
          document,
          error
        }) => {
          errors.push({
            message: (error === null || error === void 0 ? void 0 : error.reason) || 'Unknown error',
            index: document.index
          });
        }
      });
      stats.successful += successful;
      stats.failed += failed;
      return {
        errors,
        stats
      };
    });
    this.options = options;
  }
  /**
   * It will create idex for asset criticality,
   * or update mappings if index exists
   */
  async init() {
    await (0, _create_or_update_index.createOrUpdateIndex)({
      esClient: this.options.esClient,
      logger: this.options.logger,
      options: {
        index: this.getIndex(),
        mappings: (0, _common.mappingFromFieldMap)(_constants.assetCriticalityFieldMap, 'strict')
      }
    });
  }

  /**
   *
   * A general method for searching asset criticality records.
   * @param query an ESL query to filter criticality results
   * @param size the maximum number of records to return. Cannot exceed {@link MAX_CRITICALITY_RESPONSE_SIZE}. If unspecified, will default to {@link DEFAULT_CRITICALITY_RESPONSE_SIZE}.
   * @returns criticality records matching the query
   */
  async search({
    query,
    size
  }) {
    const response = await this.options.esClient.search({
      index: this.getIndex(),
      ignore_unavailable: true,
      body: {
        query
      },
      size: Math.min(size !== null && size !== void 0 ? size : DEFAULT_CRITICALITY_RESPONSE_SIZE, MAX_CRITICALITY_RESPONSE_SIZE)
    });
    return response;
  }
  getIndex() {
    return (0, _asset_criticality.getAssetCriticalityIndex)(this.options.namespace);
  }
  async doesIndexExist() {
    try {
      const result = await this.options.esClient.indices.exists({
        index: this.getIndex()
      });
      return result;
    } catch (e) {
      return false;
    }
  }
  async getStatus() {
    const isAssetCriticalityResourcesInstalled = await this.doesIndexExist();
    return {
      isAssetCriticalityResourcesInstalled
    };
  }
  async get(idParts) {
    const id = createId(idParts);
    try {
      const body = await this.options.esClient.get({
        id,
        index: this.getIndex()
      });
      return body._source;
    } catch (err) {
      if (err.statusCode === 404) {
        return undefined;
      } else {
        throw err;
      }
    }
  }
  async upsert(record) {
    const id = createId(record);
    const doc = {
      id_field: record.idField,
      id_value: record.idValue,
      criticality_level: record.criticalityLevel,
      '@timestamp': new Date().toISOString()
    };
    await this.options.esClient.update({
      id,
      index: this.getIndex(),
      body: {
        doc,
        doc_as_upsert: true
      }
    });
    return doc;
  }
  async delete(idParts) {
    await this.options.esClient.delete({
      id: createId(idParts),
      index: this.getIndex()
    });
  }
}
exports.AssetCriticalityDataClient = AssetCriticalityDataClient;