"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.groupAndBulkCreate = exports.filterBucketHistory = exports.buildBucketHistoryFilter = void 0;
var _with_security_span = require("../../../../../utils/with_security_span");
var _build_events_query = require("../../utils/build_events_query");
var _utils = require("../../utils/utils");
var _wrap_suppressed_alerts = require("./wrap_suppressed_alerts");
var _build_group_by_field_aggregation = require("./build_group_by_field_aggregation");
var _single_search_after = require("../../utils/single_search_after");
var _bulk_create_with_suppression = require("../../utils/bulk_create_with_suppression");
var _rule_schema = require("../../../../../../common/api/detection_engine/model/rule_schema");
var _bulk_create_unsuppressed_alerts = require("./bulk_create_unsuppressed_alerts");
var _constants = require("../../../../../../common/detection_engine/constants");
var _enrichments = require("../../utils/enrichments");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Builds a filter that excludes documents from existing buckets.
 */
const buildBucketHistoryFilter = ({
  bucketHistory,
  primaryTimestamp,
  secondaryTimestamp,
  from
}) => {
  if (bucketHistory.length === 0) {
    return undefined;
  }
  return [{
    bool: {
      must_not: bucketHistory.map(bucket => ({
        bool: {
          must_not: Object.entries(bucket.key).filter(([_, value]) => value == null).map(([field, _]) => ({
            exists: {
              field
            }
          })),
          filter: [...Object.entries(bucket.key).filter(([_, value]) => value != null).map(([field, value]) => ({
            term: {
              [field]: value
            }
          })), (0, _build_events_query.buildTimeRangeFilter)({
            to: bucket.endDate,
            from: from.toISOString(),
            primaryTimestamp,
            secondaryTimestamp
          })]
        }
      }))
    }
  }];
};
exports.buildBucketHistoryFilter = buildBucketHistoryFilter;
const filterBucketHistory = ({
  bucketHistory,
  fromDate
}) => {
  return bucketHistory.filter(bucket => new Date(bucket.endDate) > fromDate);
};
exports.filterBucketHistory = filterBucketHistory;
const groupAndBulkCreate = async ({
  runOpts,
  services,
  spaceId,
  filter,
  buildReasonMessage,
  bucketHistory,
  groupByFields,
  eventsTelemetry,
  experimentalFeatures
}) => {
  return (0, _with_security_span.withSecuritySpan)('groupAndBulkCreate', async () => {
    const tuple = runOpts.tuple;
    const filteredBucketHistory = filterBucketHistory({
      bucketHistory: bucketHistory !== null && bucketHistory !== void 0 ? bucketHistory : [],
      fromDate: tuple.from.toDate()
    });
    let toReturn = {
      success: true,
      warning: false,
      searchAfterTimes: [],
      enrichmentTimes: [],
      bulkCreateTimes: [],
      lastLookBackDate: null,
      createdSignalsCount: 0,
      createdSignals: [],
      errors: [],
      warningMessages: [],
      state: {
        suppressionGroupHistory: filteredBucketHistory
      }
    };
    const exceptionsWarning = (0, _utils.getUnprocessedExceptionsWarnings)(runOpts.unprocessedExceptions);
    if (exceptionsWarning) {
      toReturn.warningMessages.push(exceptionsWarning);
    }
    try {
      var _runOpts$completeRule, _runOpts$completeRule2, _runOpts$completeRule3;
      if (groupByFields.length === 0) {
        throw new Error('groupByFields length must be greater than 0');
      }
      const bucketHistoryFilter = buildBucketHistoryFilter({
        bucketHistory: filteredBucketHistory,
        primaryTimestamp: runOpts.primaryTimestamp,
        secondaryTimestamp: runOpts.secondaryTimestamp,
        from: tuple.from
      });

      // if we do not suppress alerts for docs with missing values, we will create aggregation for null missing buckets
      const suppressOnMissingFields = ((_runOpts$completeRule = (_runOpts$completeRule2 = runOpts.completeRule.ruleParams.alertSuppression) === null || _runOpts$completeRule2 === void 0 ? void 0 : _runOpts$completeRule2.missingFieldsStrategy) !== null && _runOpts$completeRule !== void 0 ? _runOpts$completeRule : _constants.DEFAULT_SUPPRESSION_MISSING_FIELDS_STRATEGY) === _rule_schema.AlertSuppressionMissingFieldsStrategyEnum.suppress;
      const groupingAggregation = (0, _build_group_by_field_aggregation.buildGroupByFieldAggregation)({
        groupByFields,
        maxSignals: tuple.maxSignals,
        aggregatableTimestampField: runOpts.aggregatableTimestampField,
        missingBucket: suppressOnMissingFields
      });
      const eventsSearchParams = {
        aggregations: groupingAggregation,
        searchAfterSortIds: undefined,
        index: runOpts.inputIndex,
        from: tuple.from.toISOString(),
        to: tuple.to.toISOString(),
        services,
        ruleExecutionLogger: runOpts.ruleExecutionLogger,
        filter,
        pageSize: 0,
        primaryTimestamp: runOpts.primaryTimestamp,
        secondaryTimestamp: runOpts.secondaryTimestamp,
        runtimeMappings: runOpts.runtimeMappings,
        additionalFilters: bucketHistoryFilter
      };
      const {
        searchResult,
        searchDuration,
        searchErrors
      } = await (0, _single_search_after.singleSearchAfter)(eventsSearchParams);
      toReturn.searchAfterTimes.push(searchDuration);
      toReturn.errors.push(...searchErrors);
      const eventsByGroupResponseWithAggs = searchResult;
      if (!eventsByGroupResponseWithAggs.aggregations) {
        throw new Error('expected to find aggregations on search result');
      }
      const buckets = eventsByGroupResponseWithAggs.aggregations.eventGroups.buckets;

      // we can create only as many unsuppressed alerts, as total number of alerts(suppressed and unsuppressed) does not exceeds maxSignals
      const maxUnsuppressedCount = tuple.maxSignals - buckets.length;
      if (suppressOnMissingFields === false && maxUnsuppressedCount > 0) {
        const unsuppressedResult = await (0, _bulk_create_unsuppressed_alerts.bulkCreateUnsuppressedAlerts)({
          groupByFields,
          size: maxUnsuppressedCount,
          runOpts,
          buildReasonMessage,
          eventsTelemetry,
          filter,
          services
        });
        toReturn = {
          ...toReturn,
          ...(0, _utils.mergeReturns)([toReturn, unsuppressedResult])
        };
      }
      if (buckets.length === 0) {
        return toReturn;
      }
      if (buckets.length > tuple.maxSignals && !toReturn.warningMessages.includes((0, _utils.getMaxSignalsWarning)()) // If the unsuppressed result didn't already hit max signals, we add the warning here
      ) {
        toReturn.warningMessages.push((0, _utils.getMaxSignalsWarning)());
      }
      const suppressionBuckets = buckets.map(bucket => ({
        event: bucket.topHits.hits.hits[0],
        count: bucket.doc_count,
        start: bucket.min_timestamp.value_as_string ? new Date(bucket.min_timestamp.value_as_string) : tuple.from.toDate(),
        end: bucket.max_timestamp.value_as_string ? new Date(bucket.max_timestamp.value_as_string) : tuple.to.toDate(),
        terms: Object.entries(bucket.key).map(([key, value]) => ({
          field: key,
          value
        }))
      }));
      const wrappedAlerts = (0, _wrap_suppressed_alerts.wrapSuppressedAlerts)({
        suppressionBuckets,
        spaceId,
        completeRule: runOpts.completeRule,
        mergeStrategy: runOpts.mergeStrategy,
        indicesToQuery: runOpts.inputIndex,
        publicBaseUrl: runOpts.publicBaseUrl,
        buildReasonMessage,
        alertTimestampOverride: runOpts.alertTimestampOverride,
        ruleExecutionLogger: runOpts.ruleExecutionLogger
      });
      const suppressionDuration = (_runOpts$completeRule3 = runOpts.completeRule.ruleParams.alertSuppression) === null || _runOpts$completeRule3 === void 0 ? void 0 : _runOpts$completeRule3.duration;
      if (suppressionDuration) {
        const suppressionWindow = `now-${suppressionDuration.value}${suppressionDuration.unit}`;
        const bulkCreateResult = await (0, _bulk_create_with_suppression.bulkCreateWithSuppression)({
          alertWithSuppression: runOpts.alertWithSuppression,
          ruleExecutionLogger: runOpts.ruleExecutionLogger,
          wrappedDocs: wrappedAlerts,
          services,
          suppressionWindow,
          alertTimestampOverride: runOpts.alertTimestampOverride,
          experimentalFeatures
        });
        (0, _utils.addToSearchAfterReturn)({
          current: toReturn,
          next: bulkCreateResult
        });
        runOpts.ruleExecutionLogger.debug(`created ${bulkCreateResult.createdItemsCount} signals`);
      } else {
        const bulkCreateResult = await runOpts.bulkCreate(wrappedAlerts, undefined, (0, _enrichments.createEnrichEventsFunction)({
          services,
          logger: runOpts.ruleExecutionLogger
        }));
        (0, _utils.addToSearchAfterReturn)({
          current: toReturn,
          next: bulkCreateResult
        });
        runOpts.ruleExecutionLogger.debug(`created ${bulkCreateResult.createdItemsCount} signals`);
      }
      const newBucketHistory = buckets.map(bucket => {
        return {
          key: bucket.key,
          endDate: bucket.max_timestamp.value_as_string ? bucket.max_timestamp.value_as_string : tuple.to.toISOString()
        };
      });
      toReturn.state.suppressionGroupHistory.push(...newBucketHistory);
    } catch (exc) {
      toReturn.success = false;
      toReturn.errors.push(exc.message);
    }
    return toReturn;
  });
};
exports.groupAndBulkCreate = groupAndBulkCreate;