"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateHealthInterval = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _rule_monitoring = require("../../../../../../common/api/detection_engine/rule_monitoring");
var _utility_types = require("../../../../../../common/utility_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_INTERVAL_PARAMETERS = {
  type: _rule_monitoring.HealthIntervalType.last_day,
  granularity: _rule_monitoring.HealthIntervalGranularity.hour
};
const validateHealthInterval = (params, now) => {
  const parameters = params !== null && params !== void 0 ? params : DEFAULT_INTERVAL_PARAMETERS;
  const from = getFrom(parameters, now);
  const to = getTo(parameters, now);
  const duration = _moment.default.duration(to.diff(from));

  // TODO: https://github.com/elastic/kibana/issues/125642 Validate that:
  //   - to > from
  //   - granularity is not too big, e.g. < duration (could be invalid when custom_range)
  //   - granularity is not too small (could be invalid when custom_range)

  return {
    type: parameters.type,
    granularity: parameters.granularity,
    from: from.utc().toISOString(),
    to: to.utc().toISOString(),
    duration: duration.toISOString()
  };
};
exports.validateHealthInterval = validateHealthInterval;
const getFrom = (params, now) => {
  const {
    type
  } = params;

  // NOTE: it's important to clone `now` with `moment(now)` because moment objects are mutable.
  // If you call .subtract() or other methods on the original `now`, you will change it which
  // might cause bugs depending on how you use it in your calculations later.

  if (type === _rule_monitoring.HealthIntervalType.custom_range) {
    return (0, _moment.default)(params.from);
  }
  if (type === _rule_monitoring.HealthIntervalType.last_hour) {
    return (0, _moment.default)(now).subtract(1, 'hour');
  }
  if (type === _rule_monitoring.HealthIntervalType.last_day) {
    return (0, _moment.default)(now).subtract(1, 'day');
  }
  if (type === _rule_monitoring.HealthIntervalType.last_week) {
    return (0, _moment.default)(now).subtract(1, 'week');
  }
  if (type === _rule_monitoring.HealthIntervalType.last_month) {
    return (0, _moment.default)(now).subtract(1, 'month');
  }
  if (type === _rule_monitoring.HealthIntervalType.last_year) {
    return (0, _moment.default)(now).subtract(1, 'year');
  }
  return (0, _utility_types.assertUnreachable)(type, 'Unhandled health interval type');
};
const getTo = (params, now) => {
  const {
    type
  } = params;
  if (type === _rule_monitoring.HealthIntervalType.custom_range) {
    return (0, _moment.default)(params.to);
  }

  // NOTE: it's important to clone `now` with `moment(now)` because moment objects are mutable. If you
  // return the original now from this method and then call .subtract() or other methods on it, it will
  // change the original now which might cause bugs depending on how you use it in your calculations later.

  return (0, _moment.default)(now);
};