"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performBulkActionRoute = exports.MAX_RULES_TO_PROCESS_TOTAL = void 0;
var _common = require("@kbn/kibana-utils-plugin/common");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _rule_management = require("../../../../../../../common/api/detection_engine/rule_management");
var _constants = require("../../../../../../../common/constants");
var _route_validation = require("../../../../../../utils/build_validation/route_validation");
var _promise_pool = require("../../../../../../utils/promise_pool");
var _route_limited_concurrency_tag = require("../../../../../../utils/route_limited_concurrency_tag");
var _authz = require("../../../../../machine_learning/authz");
var _utils = require("../../../../routes/utils");
var _duplicate_exceptions = require("../../../logic/actions/duplicate_exceptions");
var _duplicate_rule = require("../../../logic/actions/duplicate_rule");
var _bulk_edit_rules = require("../../../logic/bulk_actions/bulk_edit_rules");
var _validations = require("../../../logic/bulk_actions/validations");
var _delete_rules = require("../../../logic/crud/delete_rules");
var _get_export_by_object_ids = require("../../../logic/export/get_export_by_object_ids");
var _timeouts = require("../../timeouts");
var _bulk_actions_response = require("./bulk_actions_response");
var _bulk_enable_disable_rules = require("./bulk_enable_disable_rules");
var _fetch_rules_by_query_or_ids = require("./fetch_rules_by_query_or_ids");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_RULES_TO_PROCESS_TOTAL = exports.MAX_RULES_TO_PROCESS_TOTAL = 10000;
const MAX_ROUTE_CONCURRENCY = 5;
const performBulkActionRoute = (router, ml, logger) => {
  router.versioned.post({
    access: 'public',
    path: _constants.DETECTION_ENGINE_RULES_BULK_ACTION,
    options: {
      tags: ['access:securitySolution', (0, _route_limited_concurrency_tag.routeLimitedConcurrencyTag)(MAX_ROUTE_CONCURRENCY)],
      timeout: {
        idleSocket: _timeouts.RULE_MANAGEMENT_BULK_ACTION_SOCKET_TIMEOUT_MS
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidationWithZod)(_rule_management.PerformBulkActionRequestBody),
        query: (0, _route_validation.buildRouteValidationWithZod)(_rule_management.PerformBulkActionRequestQuery)
      }
    }
  }, async (context, request, response) => {
    const {
      body
    } = request;
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    if (body !== null && body !== void 0 && body.ids && body.ids.length > _constants.RULES_TABLE_MAX_PAGE_SIZE) {
      return siemResponse.error({
        body: `More than ${_constants.RULES_TABLE_MAX_PAGE_SIZE} ids sent for bulk edit action.`,
        statusCode: 400
      });
    }
    if (body !== null && body !== void 0 && body.ids && body.query !== undefined) {
      return siemResponse.error({
        body: `Both query and ids are sent. Define either ids or query in request payload.`,
        statusCode: 400
      });
    }
    const isDryRun = request.query.dry_run;

    // dry run is not supported for export, as it doesn't change ES state and has different response format(exported JSON file)
    if (isDryRun && body.action === _rule_management.BulkActionTypeEnum.export) {
      return siemResponse.error({
        body: `Export action doesn't support dry_run mode`,
        statusCode: 400
      });
    }
    const abortController = new AbortController();

    // subscribing to completed$, because it handles both cases when request was completed and aborted.
    // when route is finished by timeout, aborted$ is not getting fired
    request.events.completed$.subscribe(() => abortController.abort());
    try {
      var _ctx$lists;
      const ctx = await context.resolve(['core', 'securitySolution', 'alerting', 'licensing', 'lists', 'actions']);
      const rulesClient = ctx.alerting.getRulesClient();
      const exceptionsClient = (_ctx$lists = ctx.lists) === null || _ctx$lists === void 0 ? void 0 : _ctx$lists.getExceptionListClient();
      const savedObjectsClient = ctx.core.savedObjects.client;
      const actionsClient = ctx.actions.getActionsClient();
      const {
        getExporter,
        getClient
      } = ctx.core.savedObjects;
      const client = getClient({
        includedHiddenTypes: ['action']
      });
      const exporter = getExporter(client);
      const mlAuthz = (0, _authz.buildMlAuthz)({
        license: ctx.licensing.license,
        ml,
        request,
        savedObjectsClient
      });
      const query = body.query !== '' ? body.query : undefined;

      // handling this action before switch statement as bulkEditRules fetch rules within
      // rulesClient method, hence there is no need to use fetchRulesByQueryOrIds utility
      if (body.action === _rule_management.BulkActionTypeEnum.edit && !isDryRun) {
        const {
          rules,
          errors,
          skipped
        } = await (0, _bulk_edit_rules.bulkEditRules)({
          rulesClient,
          filter: query,
          ids: body.ids,
          actions: body.edit,
          mlAuthz
        });
        return (0, _bulk_actions_response.buildBulkResponse)(response, {
          updated: rules,
          skipped,
          errors
        });
      }
      const fetchRulesOutcome = await (0, _fetch_rules_by_query_or_ids.fetchRulesByQueryOrIds)({
        rulesClient,
        query,
        ids: body.ids,
        abortSignal: abortController.signal
      });
      const rules = fetchRulesOutcome.results.map(({
        result
      }) => result);
      const errors = [...fetchRulesOutcome.errors];
      let updated = [];
      let created = [];
      let deleted = [];
      switch (body.action) {
        case _rule_management.BulkActionTypeEnum.enable:
          {
            const {
              updatedRules,
              errors: bulkActionErrors
            } = await (0, _bulk_enable_disable_rules.bulkEnableDisableRules)({
              rules,
              isDryRun,
              rulesClient,
              action: 'enable',
              mlAuthz
            });
            errors.push(...bulkActionErrors);
            updated = updatedRules;
            break;
          }
        case _rule_management.BulkActionTypeEnum.disable:
          {
            const {
              updatedRules,
              errors: bulkActionErrors
            } = await (0, _bulk_enable_disable_rules.bulkEnableDisableRules)({
              rules,
              isDryRun,
              rulesClient,
              action: 'disable',
              mlAuthz
            });
            errors.push(...bulkActionErrors);
            updated = updatedRules;
            break;
          }
        case _rule_management.BulkActionTypeEnum.delete:
          {
            const bulkActionOutcome = await (0, _promise_pool.initPromisePool)({
              concurrency: _constants.MAX_RULES_TO_UPDATE_IN_PARALLEL,
              items: rules,
              executor: async rule => {
                // during dry run return early for delete, as no validations needed for this action
                if (isDryRun) {
                  return null;
                }
                await (0, _delete_rules.deleteRules)({
                  ruleId: rule.id,
                  rulesClient
                });
                return null;
              },
              abortSignal: abortController.signal
            });
            errors.push(...bulkActionOutcome.errors);
            deleted = bulkActionOutcome.results.map(({
              item
            }) => item).filter(rule => rule !== null);
            break;
          }
        case _rule_management.BulkActionTypeEnum.duplicate:
          {
            const bulkActionOutcome = await (0, _promise_pool.initPromisePool)({
              concurrency: _constants.MAX_RULES_TO_UPDATE_IN_PARALLEL,
              items: rules,
              executor: async rule => {
                await (0, _validations.validateBulkDuplicateRule)({
                  mlAuthz,
                  rule
                });

                // during dry run only validation is getting performed and rule is not saved in ES, thus return early
                if (isDryRun) {
                  return rule;
                }
                let shouldDuplicateExceptions = true;
                let shouldDuplicateExpiredExceptions = true;
                if (body.duplicate !== undefined) {
                  shouldDuplicateExceptions = body.duplicate.include_exceptions;
                  shouldDuplicateExpiredExceptions = body.duplicate.include_expired_exceptions;
                }
                const duplicateRuleToCreate = await (0, _duplicate_rule.duplicateRule)({
                  rule
                });
                const createdRule = await rulesClient.create({
                  data: duplicateRuleToCreate
                });

                // we try to create exceptions after rule created, and then update rule
                const exceptions = shouldDuplicateExceptions ? await (0, _duplicate_exceptions.duplicateExceptions)({
                  ruleId: rule.params.ruleId,
                  exceptionLists: rule.params.exceptionsList,
                  includeExpiredExceptions: shouldDuplicateExpiredExceptions,
                  exceptionsClient
                }) : [];
                const updatedRule = await rulesClient.update({
                  id: createdRule.id,
                  data: {
                    ...duplicateRuleToCreate,
                    params: {
                      ...duplicateRuleToCreate.params,
                      exceptionsList: exceptions
                    }
                  },
                  shouldIncrementRevision: () => false
                });

                // TODO: figureout why types can't return just updatedRule
                return {
                  ...createdRule,
                  ...updatedRule
                };
              },
              abortSignal: abortController.signal
            });
            errors.push(...bulkActionOutcome.errors);
            created = bulkActionOutcome.results.map(({
              result
            }) => result).filter(rule => rule !== null);
            break;
          }
        case _rule_management.BulkActionTypeEnum.export:
          {
            const exported = await (0, _get_export_by_object_ids.getExportByObjectIds)(rulesClient, exceptionsClient, rules.map(({
              params
            }) => params.ruleId), exporter, request, actionsClient);
            const responseBody = `${exported.rulesNdjson}${exported.exceptionLists}${exported.actionConnectors}${exported.exportDetails}`;
            return response.ok({
              headers: {
                'Content-Disposition': `attachment; filename="rules_export.ndjson"`,
                'Content-Type': 'application/ndjson'
              },
              body: responseBody
            });
          }

        // will be processed only when isDryRun === true
        // during dry run only validation is getting performed and rule is not saved in ES
        case _rule_management.BulkActionTypeEnum.edit:
          {
            const bulkActionOutcome = await (0, _promise_pool.initPromisePool)({
              concurrency: _constants.MAX_RULES_TO_UPDATE_IN_PARALLEL,
              items: rules,
              executor: async rule => {
                await (0, _validations.dryRunValidateBulkEditRule)({
                  mlAuthz,
                  rule,
                  edit: body.edit
                });
                return rule;
              },
              abortSignal: abortController.signal
            });
            errors.push(...bulkActionOutcome.errors);
            updated = bulkActionOutcome.results.map(({
              result
            }) => result).filter(rule => rule !== null);
            break;
          }
      }
      if (abortController.signal.aborted === true) {
        throw new _common.AbortError('Bulk action was aborted');
      }
      return (0, _bulk_actions_response.buildBulkResponse)(response, {
        updated,
        deleted,
        created,
        errors,
        isDryRun
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.performBulkActionRoute = performBulkActionRoute;