"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SentinelOneActionsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/stack-connectors-plugin/common/sentinelone/constants");
var _lodash = require("lodash");
var _common = require("../../../../../../common");
var _utils = require("../../../../utils");
var _stringify = require("../../../../utils/stringify");
var _errors = require("../errors");
var _base_response_actions_client = require("../lib/base_response_actions_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SentinelOneActionsClient extends _base_response_actions_client.ResponseActionsClientImpl {
  constructor({
    connectorActions,
    ...options
  }) {
    super(options);
    (0, _defineProperty2.default)(this, "agentType", 'sentinel_one');
    (0, _defineProperty2.default)(this, "connectorActionsClient", void 0);
    this.connectorActionsClient = connectorActions;
    connectorActions.setup(_constants.SENTINELONE_CONNECTOR_ID);
  }
  async writeActionRequestToEndpointIndex(actionRequest) {
    var _actionRequest$meta;
    const agentUUID = actionRequest.endpoint_ids[0];
    const agentDetails = await this.getAgentDetails(agentUUID);
    const doc = await super.writeActionRequestToEndpointIndex({
      ...actionRequest,
      hosts: {
        [agentUUID]: {
          name: agentDetails.computerName
        }
      },
      meta: {
        // Add common meta data
        agentUUID,
        agentId: agentDetails.id,
        hostName: agentDetails.computerName,
        ...((_actionRequest$meta = actionRequest.meta) !== null && _actionRequest$meta !== void 0 ? _actionRequest$meta : {})
      }
    });
    return doc;
  }

  /**
   * Sends actions to SentinelOne directly (via Connector)
   * @private
   */
  async sendAction(actionType, actionParams) {
    const executeOptions = {
      params: {
        subAction: actionType,
        subActionParams: actionParams
      }
    };
    this.log.debug(`calling connector actions 'execute()' for SentinelOne with:\n${(0, _stringify.stringify)(executeOptions)}`);
    const actionSendResponse = await this.connectorActionsClient.execute(executeOptions);
    if (actionSendResponse.status === 'error') {
      this.log.error((0, _stringify.stringify)(actionSendResponse));
      throw new _errors.ResponseActionsClientError(`Attempt to send [${actionType}] to SentinelOne failed: ${actionSendResponse.serviceMessage || actionSendResponse.message}`, 500, actionSendResponse);
    }
    this.log.debug(`Response:\n${(0, _stringify.stringify)(actionSendResponse)}`);
    return actionSendResponse;
  }

  /** Gets agent details directly from SentinelOne */
  async getAgentDetails(agentUUID) {
    const executeOptions = {
      params: {
        subAction: _constants.SUB_ACTION.GET_AGENTS,
        subActionParams: {
          uuid: agentUUID
        }
      }
    };
    let s1ApiResponse;
    try {
      const response = await this.connectorActionsClient.execute(executeOptions);
      this.log.debug(`Response for SentinelOne agent id [${agentUUID}] returned:\n${(0, _stringify.stringify)(response)}`);
      s1ApiResponse = response.data;
    } catch (err) {
      throw new _errors.ResponseActionsClientError(`Error while attempting to retrieve SentinelOne host with agent id [${agentUUID}]`, 500, err);
    }
    if (!s1ApiResponse || !s1ApiResponse.data[0]) {
      throw new _errors.ResponseActionsClientError(`SentinelOne agent id [${agentUUID}] not found`, 404);
    }
    return s1ApiResponse.data[0];
  }
  async validateRequest(payload) {
    // TODO:PT support multiple agents
    if (payload.endpoint_ids.length > 1) {
      return {
        isValid: false,
        error: new _errors.ResponseActionsClientError(`[body.endpoint_ids]: Multiple agents IDs not currently supported for SentinelOne`, 400)
      };
    }
    return super.validateRequest(payload);
  }
  async isolate(actionRequest, options = {}) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      command: 'isolate'
    };
    if (!reqIndexOptions.error) {
      var _error;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          await this.sendAction(_constants.SUB_ACTION.ISOLATE_HOST, {
            uuid: actionRequest.endpoint_ids[0]
          });
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error = error) === null || _error === void 0 ? void 0 : _error.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const actionRequestDoc = await this.writeActionRequestToEndpointIndex(reqIndexOptions);
    await this.updateCases({
      command: reqIndexOptions.command,
      caseIds: reqIndexOptions.case_ids,
      alertIds: reqIndexOptions.alert_ids,
      actionId: actionRequestDoc.EndpointActions.action_id,
      hosts: actionRequest.endpoint_ids.map(agentId => {
        var _actionRequestDoc$End, _actionRequestDoc$End2;
        return {
          hostId: agentId,
          hostname: (_actionRequestDoc$End = (_actionRequestDoc$End2 = actionRequestDoc.EndpointActions.data.hosts) === null || _actionRequestDoc$End2 === void 0 ? void 0 : _actionRequestDoc$End2[agentId].name) !== null && _actionRequestDoc$End !== void 0 ? _actionRequestDoc$End : ''
        };
      }),
      comment: reqIndexOptions.comment
    });
    if (!actionRequestDoc.error && !this.options.endpointService.experimentalFeatures.responseActionsSentinelOneV2Enabled) {
      await this.writeActionResponseToEndpointIndex({
        actionId: actionRequestDoc.EndpointActions.action_id,
        agentId: actionRequestDoc.agent.id,
        data: {
          command: actionRequestDoc.EndpointActions.data.command
        }
      });
    }
    return this.fetchActionDetails(actionRequestDoc.EndpointActions.action_id);
  }
  async release(actionRequest, options = {}) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      command: 'unisolate'
    };
    if (!reqIndexOptions.error) {
      var _error2;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          await this.sendAction(_constants.SUB_ACTION.RELEASE_HOST, {
            uuid: actionRequest.endpoint_ids[0]
          });
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error2 = error) === null || _error2 === void 0 ? void 0 : _error2.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const actionRequestDoc = await this.writeActionRequestToEndpointIndex(reqIndexOptions);
    await this.updateCases({
      command: reqIndexOptions.command,
      caseIds: reqIndexOptions.case_ids,
      alertIds: reqIndexOptions.alert_ids,
      actionId: actionRequestDoc.EndpointActions.action_id,
      hosts: actionRequest.endpoint_ids.map(agentId => {
        var _actionRequestDoc$End3, _actionRequestDoc$End4;
        return {
          hostId: agentId,
          hostname: (_actionRequestDoc$End3 = (_actionRequestDoc$End4 = actionRequestDoc.EndpointActions.data.hosts) === null || _actionRequestDoc$End4 === void 0 ? void 0 : _actionRequestDoc$End4[agentId].name) !== null && _actionRequestDoc$End3 !== void 0 ? _actionRequestDoc$End3 : ''
        };
      }),
      comment: reqIndexOptions.comment
    });
    if (!actionRequestDoc.error && !this.options.endpointService.experimentalFeatures.responseActionsSentinelOneV2Enabled) {
      await this.writeActionResponseToEndpointIndex({
        actionId: actionRequestDoc.EndpointActions.action_id,
        agentId: actionRequestDoc.agent.id,
        data: {
          command: actionRequestDoc.EndpointActions.data.command
        }
      });
    }
    return this.fetchActionDetails(actionRequestDoc.EndpointActions.action_id);
  }
  async processPendingActions({
    abortSignal,
    addToQueue
  }) {
    if (abortSignal.aborted) {
      return;
    }
    for await (const pendingActions of this.fetchAllPendingActions()) {
      if (abortSignal.aborted) {
        return;
      }
      const pendingActionsByType = (0, _lodash.groupBy)(pendingActions, 'EndpointActions.data.command');
      for (const [actionType, typePendingActions] of Object.entries(pendingActionsByType)) {
        if (abortSignal.aborted) {
          return;
        }
        switch (actionType) {
          case 'isolate':
          case 'unisolate':
            {
              const isolationResponseDocs = await this.checkPendingIsolateOrReleaseActions(typePendingActions, actionType);
              if (isolationResponseDocs.length) {
                addToQueue(...isolationResponseDocs);
              }
            }
            break;
        }
      }
    }
  }

  /**
   * Checks if the provided Isolate or Unisolate actions are complete and if so, then it builds the Response
   * document for them and returns it. (NOTE: the response is NOT written to ES - only returned)
   * @param actionRequests
   * @param command
   * @private
   */
  async checkPendingIsolateOrReleaseActions(actionRequests, command) {
    const completedResponses = [];
    const actionsByAgentId = {};
    const warnings = [];

    // Create the `OR` clause that filters for each agent id and an updated date of greater than the date when
    // the isolate request was created
    const agentListQuery = actionRequests.reduce((acc, action) => {
      var _action$meta;
      const s1AgentId = (_action$meta = action.meta) === null || _action$meta === void 0 ? void 0 : _action$meta.agentId;
      if (s1AgentId) {
        if (!actionsByAgentId[s1AgentId]) {
          actionsByAgentId[s1AgentId] = [];
        }
        actionsByAgentId[s1AgentId].push(action);
        acc.push({
          bool: {
            filter: [{
              term: {
                'sentinel_one.activity.agent.id': s1AgentId
              }
            }, {
              range: {
                'sentinel_one.activity.updated_at': {
                  gt: action['@timestamp']
                }
              }
            }]
          }
        });
      } else {
        // This is an edge case and should never happen. But just in case :-)
        warnings.push(`${command} response action ID [${action.EndpointActions.action_id}] missing SentinelOne agent ID, thus unable to check on it's status. Forcing it to complete as failure.`);
        completedResponses.push(this.buildActionResponseEsDoc({
          actionId: action.EndpointActions.action_id,
          agentId: Array.isArray(action.agent.id) ? action.agent.id[0] : action.agent.id,
          data: {
            command
          },
          error: {
            message: `Unable to very if action completed. SentinelOne agent id ('meta.agentId') missing on action request document!`
          }
        }));
      }
      return acc;
    }, []);
    if (agentListQuery.length > 0) {
      const query = {
        bool: {
          must: [{
            terms: {
              // Activity Types can be retrieved from S1 via API: `/web/api/v2.1/activities/types`
              'sentinel_one.activity.type': command === 'isolate' ? [
              // {
              //    "id": 1001
              //    "action": "Agent Disconnected From Network",
              //    "descriptionTemplate": "Agent {{ computer_name }} was disconnected from network.",
              // },
              1001,
              // {
              //    "id": 2010
              //    "action": "Agent Mitigation Report Quarantine Network Failed",
              //    "descriptionTemplate": "Agent {{ computer_name }} was unable to disconnect from network.",
              // },
              2010] : [
              // {
              //    "id": 1002
              //    "action": "Agent Reconnected To Network",
              //    "descriptionTemplate": "Agent {{ computer_name }} was connected to network.",
              // },
              1002]
            }
          }],
          should: agentListQuery,
          minimum_should_match: 1
        }
      };
      const searchRequestOptions = {
        index: _common.SENTINEL_ONE_ACTIVITY_INDEX_PATTERN,
        query,
        // There may be many documents for each host/agent, so we collapse it and only get back the
        // first one that came in after the isolate request was sent
        collapse: {
          field: 'sentinel_one.activity.agent.id',
          inner_hits: {
            name: 'first_found',
            size: 1,
            sort: [{
              'sentinel_one.activity.updated_at': 'asc'
            }]
          }
        },
        // we don't need the source. The document will be stored in `inner_hits.first_found`
        // due to use of `collapse
        _source: false,
        sort: [{
          'sentinel_one.activity.updated_at': {
            order: 'asc'
          }
        }],
        size: 1000
      };
      this.log.debug(`searching for ${command} responses from [${_common.SENTINEL_ONE_ACTIVITY_INDEX_PATTERN}] index with:\n${(0, _stringify.stringify)(searchRequestOptions, 15)}`);
      const searchResults = await this.options.esClient.search(searchRequestOptions).catch(_utils.catchAndWrapError);
      this.log.debug(`Search results for SentinelOne ${command} activity documents:\n${(0, _stringify.stringify)(searchResults)}`);
      for (const searchResultHit of searchResults.hits.hits) {
        var _searchResultHit$inne;
        const isolateActivityResponseDoc = (_searchResultHit$inne = searchResultHit.inner_hits) === null || _searchResultHit$inne === void 0 ? void 0 : _searchResultHit$inne.first_found.hits.hits[0];
        if (isolateActivityResponseDoc && isolateActivityResponseDoc._source) {
          const s1ActivityData = isolateActivityResponseDoc._source.sentinel_one.activity;
          const elasticDocId = isolateActivityResponseDoc._id;
          const s1AgentId = s1ActivityData.agent.id;
          const activityLogEntryId = s1ActivityData.id;
          const activityLogEntryType = s1ActivityData.type;
          const activityLogEntryDescription = s1ActivityData.description.primary;
          for (const actionRequest of actionsByAgentId[s1AgentId]) {
            completedResponses.push(this.buildActionResponseEsDoc({
              actionId: actionRequest.EndpointActions.action_id,
              agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
              data: {
                command
              },
              error: activityLogEntryType === 2010 && command === 'isolate' ? {
                message: activityLogEntryDescription !== null && activityLogEntryDescription !== void 0 ? activityLogEntryDescription : `Action failed. SentinelOne activity log entry [${activityLogEntryId}] has a 'type' value of 2010 indicating a failure to disconnect`
              } : undefined,
              meta: {
                elasticDocId,
                activityLogEntryId,
                activityLogEntryType,
                activityLogEntryDescription
              }
            }));
          }
        }
      }
    } else {
      this.log.debug(`Nothing to search for. List of agents IDs is empty.`);
    }
    this.log.debug(`${completedResponses.length} ${command} action responses generated:\n${(0, _stringify.stringify)(completedResponses)}`);
    if (warnings.length > 0) {
      this.log.warn(warnings.join('\n'));
    }
    return completedResponses;
  }
}
exports.SentinelOneActionsClient = SentinelOneActionsClient;