"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NormalizedExternalConnectorClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _stringify = require("../../../../utils/stringify");
var _errors = require("../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable @typescript-eslint/no-explicit-any */

/**
 * Handles setting up the usage of Stack Connectors for Response Actions and normalizes usage of
 * Connector's Sub-Actions plugin between the `ActionsClient` and the `IUnsecuredActionsClient`
 * client interfaces. It also provides better typing support.
 */
class NormalizedExternalConnectorClient {
  constructor(connectorsClient, log, options) {
    (0, _defineProperty2.default)(this, "connectorTypeId", void 0);
    (0, _defineProperty2.default)(this, "getConnectorInstance", (0, _lodash.once)(async () => {
      this.ensureSetupDone();
      let connectorList = [];
      const connectorTypeId = this.connectorTypeId;
      try {
        connectorList = await this.getAll();
      } catch (err) {
        throw new _errors.ResponseActionsClientError(`Unable to retrieve list of stack connectors in order to find one for [${connectorTypeId}]: ${err.message}`,
        // failure here is likely due to Authz, but because we don't have a good way to determine that,
        // the `statusCode` below is set to `400` instead of `401`.
        400, err);
      }
      const connector = connectorList.find(({
        actionTypeId,
        isDeprecated,
        isMissingSecrets
      }) => {
        return actionTypeId === connectorTypeId && !isDeprecated && !isMissingSecrets;
      });
      if (!connector) {
        this.log.debug((0, _stringify.stringify)(connectorList));
        throw new _errors.ResponseActionsConnectorNotConfiguredError(connectorTypeId);
      }
      this.log.debug(`Using [${this.connectorTypeId}] stack connector: "${connector.name}" (ID: ${connector.id})`);
      return connector;
    }));
    this.connectorsClient = connectorsClient;
    this.log = log;
    this.options = options;
  }
  ensureSetupDone() {
    if (!this.connectorTypeId) {
      throw new _errors.ResponseActionsClientError(`Instance has not been .setup()!`);
    }
  }
  isUnsecuredActionsClient(client) {
    // The methods below only exist in the normal `ActionsClient`
    return !('create' in client) && !('delete' in client) && !('update' in client);
  }

  /**
   * Sets up the class instance for use. Must be called prior to using methods of this class
   * @param connectorTypeId
   */
  setup(connectorTypeId) {
    if (this.connectorTypeId) {
      throw new _errors.ResponseActionsClientError(`setup() has already been called with connector [${connectorTypeId}]`);
    }
    this.connectorTypeId = connectorTypeId;
  }
  async execute({
    spaceId = 'default',
    params
  }) {
    this.ensureSetupDone();
    const {
      id: connectorId
    } = await this.getConnectorInstance();
    if (this.isUnsecuredActionsClient(this.connectorsClient)) {
      var _this$options;
      return this.connectorsClient.execute({
        requesterId: 'background_task',
        id: connectorId,
        spaceId,
        params,
        relatedSavedObjects: (_this$options = this.options) === null || _this$options === void 0 ? void 0 : _this$options.relatedSavedObjects
      });
    }
    return this.connectorsClient.execute({
      actionId: connectorId,
      params
    });
  }
  async getAll(spaceId = 'default') {
    this.ensureSetupDone();
    if (this.isUnsecuredActionsClient(this.connectorsClient)) {
      return this.connectorsClient.getAll(spaceId);
    }
    return this.connectorsClient.getAll();
  }
}
exports.NormalizedExternalConnectorClient = NormalizedExternalConnectorClient;