"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAgentStatusRoute = exports.getAgentStatusRouteHandler = void 0;
var _agent_status = require("../../services/agent/agent_status");
var _error_handler = require("../error_handler");
var _get_agent_status_route = require("../../../../common/api/endpoint/agent/get_agent_status_route");
var _constants = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerAgentStatusRoute = (router, endpointContext) => {
  router.versioned.get({
    access: 'internal',
    path: _constants.ENDPOINT_AGENT_STATUS_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '1',
    validate: {
      request: _get_agent_status_route.EndpointAgentStatusRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canReadSecuritySolution']
  }, endpointContext.logFactory.get('actionStatusRoute'), getAgentStatusRouteHandler(endpointContext)));
};
exports.registerAgentStatusRoute = registerAgentStatusRoute;
const getAgentStatusRouteHandler = endpointContext => {
  const logger = endpointContext.logFactory.get('agentStatus');
  return async (context, request, response) => {
    const {
      agentType = 'endpoint',
      agentIds: _agentIds
    } = request.query;
    const agentIds = Array.isArray(_agentIds) ? _agentIds : [_agentIds];

    // Note:  because our API schemas are defined as module static variables (as opposed to a
    //        `getter` function), we need to include this additional validation here, since
    //        `agent_type` is included in the schema independent of the feature flag
    if (agentType === 'sentinel_one' && !endpointContext.experimentalFeatures.responseActionsSentinelOneV1Enabled) {
      return (0, _error_handler.errorHandler)(logger, response, new _custom_http_request_error.CustomHttpRequestError(`[request query.agent_type]: feature is disabled`, 400));
    }

    // TEMPORARY:
    // For v8.13 we only support SentinelOne on this API due to time constraints
    if (agentType !== 'sentinel_one') {
      return (0, _error_handler.errorHandler)(logger, response, new _custom_http_request_error.CustomHttpRequestError(`[${agentType}] agent type is not currently supported by this API`, 400));
    }
    logger.debug(`Retrieving status for: agentType [${agentType}], agentIds: [${agentIds.join(', ')}]`);
    try {
      return response.ok({
        body: {
          data: await (0, _agent_status.getAgentStatus)({
            agentType,
            agentIds,
            logger,
            connectorActionsClient: (await context.actions).getActionsClient()
          })
        }
      });
    } catch (e) {
      return (0, _error_handler.errorHandler)(logger, response, e);
    }
  };
};
exports.getAgentStatusRouteHandler = getAgentStatusRouteHandler;