"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addPinnedEventToTimelineMiddleware = void 0;
var _fp = require("lodash/fp");
var _selectors = require("../selectors");
var i18n = _interopRequireWildcard(require("../../pages/translations"));
var _actions = require("../actions");
var _api = require("../../containers/pinned_event/api");
var _helpers = require("./helpers");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const timelinePinnedEventActionsType = new Set([_actions.pinEvent.type, _actions.unPinEvent.type]);
function isPinnedEventAction(action) {
  return timelinePinnedEventActionsType.has(action.type);
}
const addPinnedEventToTimelineMiddleware = kibana => store => next => async action => {
  // perform the action
  const ret = next(action);
  if (isPinnedEventAction(action)) {
    const {
      id: localTimelineId,
      eventId
    } = action.payload;
    store.dispatch((0, _actions.startTimelineSaving)({
      id: localTimelineId
    }));
    try {
      // In case an event is pinned on an unsaved timeline, we need to make sure
      // the timeline has been saved or is in draft state. Otherwise, `timelineId` will be `null`
      // and we're creating orphaned pinned events.
      const timeline = await (0, _helpers.ensureTimelineIsSaved)({
        localTimelineId,
        timeline: (0, _selectors.selectTimelineById)(store.getState(), localTimelineId),
        store
      });
      if (!timeline.savedObjectId) {
        throw new Error('Cannot create a pinned event without a timelineId');
      }
      const result = await (0, _api.persistPinnedEvent)({
        pinnedEventId: timeline.pinnedEventsSaveObject[eventId] != null ? timeline.pinnedEventsSaveObject[eventId].pinnedEventId : null,
        eventId,
        timelineId: timeline.savedObjectId
      });
      const response = (0, _fp.get)('data.persistPinnedEventOnTimeline', result);
      if (response && response.code === 403) {
        store.dispatch((0, _actions.showCallOutUnauthorizedMsg)());
      }
      (0, _helpers.refreshTimelines)(store.getState());
      const currentTimeline = (0, _selectors.selectTimelineById)(store.getState(), action.payload.id);
      // The response is null in case we unpinned an event.
      // In that case we want to remove the locally pinned event.
      if (!response) {
        return store.dispatch((0, _actions.updateTimeline)({
          id: action.payload.id,
          timeline: {
            ...currentTimeline,
            pinnedEventIds: (0, _fp.omit)(eventId, currentTimeline.pinnedEventIds),
            pinnedEventsSaveObject: (0, _fp.omit)(eventId, currentTimeline.pinnedEventsSaveObject)
          }
        }));
      } else {
        const updatedTimeline = {
          ...currentTimeline,
          pinnedEventIds: {
            ...currentTimeline.pinnedEventIds,
            [eventId]: true
          },
          pinnedEventsSaveObject: {
            ...currentTimeline.pinnedEventsSaveObject,
            [eventId]: response
          }
        };
        await store.dispatch((0, _actions.updateTimeline)({
          id: action.payload.id,
          timeline: updatedTimeline
        }));
      }
    } catch (error) {
      var _error$message;
      kibana.notifications.toasts.addDanger({
        title: i18n.UPDATE_TIMELINE_ERROR_TITLE,
        text: (_error$message = error === null || error === void 0 ? void 0 : error.message) !== null && _error$message !== void 0 ? _error$message : i18n.UPDATE_TIMELINE_ERROR_TEXT
      });
    } finally {
      store.dispatch((0, _actions.endTimelineSaving)({
        id: localTimelineId
      }));
    }
  }
  return ret;
};
exports.addPinnedEventToTimelineMiddleware = addPinnedEventToTimelineMiddleware;