"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.visibleNodesAndEdgeLines = exports.userIsPanning = exports.treeRequestParametersToAbort = exports.treeParametersToFetch = exports.translation = exports.totalRelatedEventCountForNode = exports.timeRangeFilters = exports.statsTotalForNode = exports.selectedNode = exports.selectAnalyzerById = exports.selectAnalyzer = exports.scalingFactor = exports.scale = exports.resolverTreeSourceAndSchema = exports.resolverTreeHasNodes = exports.resolverComponentInstanceID = exports.relativeHref = exports.relatedEventTotalCount = exports.relatedEventCountOfTypeForNode = exports.projectionMatrix = exports.panelViewAndParameters = exports.overriddenTimeBounds = exports.originID = exports.nodeStats = exports.nodeEventsInCategory = exports.nodeDataStatus = exports.nodeDataForID = exports.newIDsToRequest = exports.layout = exports.lastRelatedEventResponseContainsCursor = exports.isTreeLoading = exports.isLoadingNodeEventsInCategory = exports.isLoadingMoreNodeEventsInCategory = exports.isCurrentRelatedEventLoading = exports.isAnimating = exports.inverseProjectionMatrix = exports.hasMoreGenerations = exports.hasMoreChildren = exports.hasMoreAncestors = exports.hadErrorLoadingTree = exports.hadErrorLoadingNodeEventsInCategory = exports.graphNodeForID = exports.eventIndices = exports.detectedBounds = exports.currentRelatedEventData = exports.currentAppliedTimeRange = exports.ariaLevel = exports.ariaFlowtoNodeID = exports.ariaActiveDescendant = exports.analyzerByIdSelector = void 0;
var _reselect = require("reselect");
var cameraSelectors = _interopRequireWildcard(require("./camera/selectors"));
var dataSelectors = _interopRequireWildcard(require("./data/selectors"));
var uiSelectors = _interopRequireWildcard(require("./ui/selectors"));
var nodeModel = _interopRequireWildcard(require("../../../common/endpoint/models/node"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const selectAnalyzer = state => state.analyzer;
exports.selectAnalyzer = selectAnalyzer;
const selectAnalyzerById = (state, id) => state.analyzer[id];
exports.selectAnalyzerById = selectAnalyzerById;
const analyzerByIdSelector = exports.analyzerByIdSelector = (0, _reselect.createSelector)(selectAnalyzer, analyzer => analyzer);

/**
 * A matrix that when applied to a Vector2 will convert it from world coordinates to screen coordinates.
 * See https://en.wikipedia.org/wiki/Orthographic_projection
 */
const projectionMatrix = exports.projectionMatrix = composeSelectors(cameraStateSelector, cameraSelectors.projectionMatrix);
const translation = exports.translation = composeSelectors(cameraStateSelector, cameraSelectors.translation);
const detectedBounds = exports.detectedBounds = composeSelectors(dataStateSelector, dataSelectors.detectedBounds);
const overriddenTimeBounds = exports.overriddenTimeBounds = composeSelectors(dataStateSelector, dataSelectors.overriddenTimeBounds);
const currentAppliedTimeRange = exports.currentAppliedTimeRange = composeSelectors(dataStateSelector, dataSelectors.currentAppliedTimeRange);

/**
 * A matrix that when applied to a Vector2 converts it from screen coordinates to world coordinates.
 * See https://en.wikipedia.org/wiki/Orthographic_projection
 */
const inverseProjectionMatrix = exports.inverseProjectionMatrix = composeSelectors(cameraStateSelector, cameraSelectors.inverseProjectionMatrix);

/**
 * The scale by which world values are scaled when rendered.
 */
const scale = exports.scale = composeSelectors(cameraStateSelector, cameraSelectors.scale);

/**
 * Scales the coordinate system, used for zooming. Should always be between 0 and 1
 */
const scalingFactor = exports.scalingFactor = composeSelectors(cameraStateSelector, cameraSelectors.scalingFactor);

/**
 * Whether or not the user is current panning the map.
 */
const userIsPanning = exports.userIsPanning = composeSelectors(cameraStateSelector, cameraSelectors.userIsPanning);

/**
 * Whether or not the camera is animating, at a given time.
 */
const isAnimating = exports.isAnimating = composeSelectors(cameraStateSelector, cameraSelectors.isAnimating);
const resolverTreeHasNodes = exports.resolverTreeHasNodes = composeSelectors(dataStateSelector, dataSelectors.resolverTreeHasNodes);

/**
 * The position of nodes and edges.
 */
const layout = exports.layout = composeSelectors(dataStateSelector, dataSelectors.layout);

/**
 * If we need to fetch, this is the entity ID to fetch.
 */
const treeParametersToFetch = exports.treeParametersToFetch = composeSelectors(dataStateSelector, dataSelectors.treeParametersToFetch);
const treeRequestParametersToAbort = exports.treeRequestParametersToAbort = composeSelectors(dataStateSelector, dataSelectors.treeRequestParametersToAbort);

/**
 * An array of indices to use for resolver panel requests.
 */
const eventIndices = exports.eventIndices = composeSelectors(dataStateSelector, dataSelectors.eventIndices);
const resolverComponentInstanceID = exports.resolverComponentInstanceID = composeSelectors(dataStateSelector, dataSelectors.resolverComponentInstanceID);

/**
 * This returns a map of nodeIDs to the associated stats provided by the datasource.
 */
const nodeStats = exports.nodeStats = composeSelectors(dataStateSelector, dataSelectors.nodeStats);

/**
 * This returns the "aggregate total" for related events, tallied as the sum
 * of their individual `event.category`s. E.g. a [DNS, Network] would count as two
 * towards the aggregate total.
 */
const relatedEventTotalCount = exports.relatedEventTotalCount = composeSelectors(dataStateSelector, dataSelectors.relatedEventTotalCount);

/**
 * the loading state of the current related event data for the `event_detail` view
 */
const isCurrentRelatedEventLoading = exports.isCurrentRelatedEventLoading = composeSelectors(dataStateSelector, dataSelectors.isCurrentRelatedEventLoading);

/**
 * the current related event data for the `event_detail` view
 */
const currentRelatedEventData = exports.currentRelatedEventData = composeSelectors(dataStateSelector, dataSelectors.currentRelatedEventData);
const timeRangeFilters = exports.timeRangeFilters = composeSelectors(dataStateSelector, dataSelectors.timeRangeFilters);

/**
 * Returns the id of the "current" tree node (fake-focused)
 */
const ariaActiveDescendant = exports.ariaActiveDescendant = composeSelectors(uiStateSelector, uiSelectors.ariaActiveDescendant);

/**
 * Returns the nodeID of the selected node
 */
const selectedNode = exports.selectedNode = composeSelectors(uiStateSelector, uiSelectors.selectedNode);

/**
 * Returns the camera state from within ResolverState
 */
function cameraStateSelector(state) {
  return state.camera;
}

/**
 * Returns the data state from within ResolverState
 */
function dataStateSelector(state) {
  return state.data;
}

/**
 * Returns the ui state from within ResolverState
 */
function uiStateSelector(state) {
  return state.ui;
}

/**
 * Whether or not the resolver is pending fetching data
 */
const isTreeLoading = exports.isTreeLoading = composeSelectors(dataStateSelector, dataSelectors.isTreeLoading);

/**
 * Whether or not the resolver encountered an error while fetching data
 */
const hadErrorLoadingTree = exports.hadErrorLoadingTree = composeSelectors(dataStateSelector, dataSelectors.hadErrorLoadingTree);

/**
 * True there might be more descendants to retrieve in the resolver graph.
 */
const hasMoreChildren = exports.hasMoreChildren = composeSelectors(dataStateSelector, dataSelectors.hasMoreChildren);

/**
 * True if there might be more ancestors to retrieve in the resolver graph.
 */
const hasMoreAncestors = exports.hasMoreAncestors = composeSelectors(dataStateSelector, dataSelectors.hasMoreAncestors);

/**
 * True if there might be more generations to retrieve in the resolver graph.
 */
const hasMoreGenerations = exports.hasMoreGenerations = composeSelectors(dataStateSelector, dataSelectors.hasMoreGenerations);
const boundingBox = composeSelectors(cameraStateSelector, cameraSelectors.viewableBoundingBox);
const nodesAndEdgelines = composeSelectors(dataStateSelector, dataSelectors.nodesAndEdgelines);

/**
 * Total count of related events for a process.
 * @deprecated
 */
const statsTotalForNode = exports.statsTotalForNode = composeSelectors(dataStateSelector, dataSelectors.statsTotalForNode);

/**
 * Return the visible edge lines and process nodes based on the camera position at `time`.
 * The bounding box represents what the camera can see. The camera position is a function of time because it can be
 * animated. So in order to get the currently visible entities, we need to pass in time.
 */
const visibleNodesAndEdgeLines = exports.visibleNodesAndEdgeLines = (0, _reselect.createSelector)(nodesAndEdgelines, boundingBox, function (nodesAndEdgelinesFn, boundingBoxFn) {
  // `boundingBox` and `nodesAndEdgelines` are each memoized.
  return time => nodesAndEdgelinesFn(boundingBoxFn(time));
});

/**
 * Takes a nodeID (aka entity_id) and returns the associated aria level as a number or null if the node ID isn't in the tree.
 */
const ariaLevel = exports.ariaLevel = composeSelectors(dataStateSelector, dataSelectors.ariaLevel);

/**
 * the node ID of the node representing the databaseDocumentID
 */
const originID = exports.originID = composeSelectors(dataStateSelector, dataSelectors.originID);

/**
 * Takes a nodeID (aka entity_id) and returns the node ID of the node that aria should 'flowto' or null
 * If the node has a flowto candidate that is currently visible, that will be returned, otherwise null.
 */
const ariaFlowtoNodeID = exports.ariaFlowtoNodeID = (0, _reselect.createSelector)(visibleNodesAndEdgeLines, composeSelectors(dataStateSelector, dataSelectors.ariaFlowtoCandidate), function (visibleNodesAndEdgeLinesAtTime, ariaFlowtoCandidate) {
  return (0, _reselect.defaultMemoize)(time => {
    // get the visible nodes at `time`
    const {
      processNodePositions
    } = visibleNodesAndEdgeLinesAtTime(time);

    // get a `Set` containing their node IDs
    const nodesVisibleAtTime = new Set();
    // NB: in practice, any event that has been graphed is guaranteed to have an entity_id
    for (const visibleNode of processNodePositions.keys()) {
      const nodeID = nodeModel.nodeID(visibleNode);
      if (nodeID !== undefined) {
        nodesVisibleAtTime.add(nodeID);
      }
    }

    // return the ID of `nodeID`'s following sibling, if it is visible
    return nodeID => {
      const flowtoNode = ariaFlowtoCandidate(nodeID);
      return flowtoNode === null || nodesVisibleAtTime.has(flowtoNode) === false ? null : flowtoNode;
    };
  });
});
const panelViewAndParameters = exports.panelViewAndParameters = composeSelectors(uiStateSelector, uiSelectors.panelViewAndParameters);
const relativeHref = exports.relativeHref = composeSelectors(uiStateSelector, uiSelectors.relativeHref);

/**
 * Total count of events related to `nodeID`.
 * Based on `ResolverNodeStats`
 */
const totalRelatedEventCountForNode = exports.totalRelatedEventCountForNode = composeSelectors(dataStateSelector, dataSelectors.totalRelatedEventCountForNode);

/**
 * Count of events with `category` related to `nodeID`.
 * Based on `ResolverNodeStats`
 * Used to populate the breadcrumbs in the `nodeEventsInCategory` panel.
 */
const relatedEventCountOfTypeForNode = exports.relatedEventCountOfTypeForNode = composeSelectors(dataStateSelector, dataSelectors.relatedEventCountOfTypeForNode);

/**
 * Events related to the panel node that are in the panel category.
 * Used to populate the breadcrumbs in the `nodeEventsInCategory` panel.
 * NB: This cannot tell the view loading information. For example, this does not tell the view if data has been request or if data failed to load.
 */
const nodeEventsInCategory = exports.nodeEventsInCategory = composeSelectors(dataStateSelector, dataSelectors.nodeEventsInCategory);

/**
 * Flag used to show a Load More Data button in the nodeEventsOfType panel view.
 */
const lastRelatedEventResponseContainsCursor = exports.lastRelatedEventResponseContainsCursor = composeSelectors(dataStateSelector, dataSelectors.lastRelatedEventResponseContainsCursor);

/**
 * Flag to show an error message when loading more related events.
 */
const hadErrorLoadingNodeEventsInCategory = exports.hadErrorLoadingNodeEventsInCategory = composeSelectors(dataStateSelector, dataSelectors.hadErrorLoadingNodeEventsInCategory);
/**
 * Flag used to show a loading view for the initial loading of related events.
 */
const isLoadingNodeEventsInCategory = exports.isLoadingNodeEventsInCategory = composeSelectors(dataStateSelector, dataSelectors.isLoadingNodeEventsInCategory);

/**
 * Flag used to show a loading state for any additional related events.
 */
const isLoadingMoreNodeEventsInCategory = exports.isLoadingMoreNodeEventsInCategory = composeSelectors(dataStateSelector, dataSelectors.isLoadingMoreNodeEventsInCategory);

/**
 * Returns the state of the node, loading, running, or terminated.
 */
const nodeDataStatus = exports.nodeDataStatus = composeSelectors(dataStateSelector, dataSelectors.nodeDataStatus);

/**
 * Returns the node data object for a specific node ID.
 */
const nodeDataForID = exports.nodeDataForID = composeSelectors(dataStateSelector, dataSelectors.nodeDataForID);

/**
 * Returns the graph node for a given ID
 */
const graphNodeForID = exports.graphNodeForID = composeSelectors(dataStateSelector, dataSelectors.graphNodeForID);

/**
 * Returns a Set of node IDs representing the visible nodes in the view that we do no have node data for already.
 */
const newIDsToRequest = exports.newIDsToRequest = (0, _reselect.createSelector)(composeSelectors(dataStateSelector, dataState => dataState.nodeData), visibleNodesAndEdgeLines, function (nodeData, visibleNodesAndEdgeLinesAtTime) {
  return (0, _reselect.defaultMemoize)(time => {
    const {
      processNodePositions: nodesInView
    } = visibleNodesAndEdgeLinesAtTime(time);
    const nodes = new Set();
    // loop through the nodes in view and see if any of them are new aka we don't have node data for them already
    for (const node of nodesInView.keys()) {
      const id = nodeModel.nodeID(node);
      // if the node has a valid ID field, and we either don't have any node data currently, or
      // the map doesn't have info for this particular node, then add it to the set so it'll be requested
      // by the middleware
      if (id !== undefined && (!nodeData || !nodeData.has(id))) {
        nodes.add(id);
      }
    }
    return nodes;
  });
});

/**
 * Returns the schema for the current resolver tree. Currently, only used in the graph controls panel.
 */
const resolverTreeSourceAndSchema = exports.resolverTreeSourceAndSchema = composeSelectors(dataStateSelector, dataSelectors.resolverTreeSourceAndSchema);

/**
 * Calls the `secondSelector` with the result of the `selector`. Use this when re-exporting a
 * concern-specific selector. `selector` should return the concern-specific state.
 */
function composeSelectors(selector, secondSelector) {
  return state => secondSelector(selector(state));
}