"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResolverTreeFetcher = ResolverTreeFetcher;
var selectors = _interopRequireWildcard(require("../selectors"));
var _ecs_safety_helpers = require("../../../../common/endpoint/models/ecs_safety_helpers");
var _resolver_tree = require("../../models/resolver_tree");
var _action = require("../data/action");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * A function that handles syncing ResolverTree data w/ the current entity ID.
 * This will make a request anytime the entityID changes (to something other than undefined.)
 * If the entity ID changes while a request is in progress, the in-progress request will be cancelled.
 * Call the returned function after each state transition.
 * This is a factory because it is stateful and keeps that state in closure.
 */
function ResolverTreeFetcher(dataAccessLayer, api) {
  let lastRequestAbortController;
  // Call this after each state change.
  // This fetches the ResolverTree for the current entityID
  // if the entityID changes while
  return async id => {
    // const id = 'alerts-page';
    const state = api.getState();
    const databaseParameters = selectors.treeParametersToFetch(state.analyzer[id]);
    if (selectors.treeRequestParametersToAbort(state.analyzer[id]) && lastRequestAbortController) {
      lastRequestAbortController.abort();
      // calling abort will cause an action to be fired
    } else if (databaseParameters !== null) {
      lastRequestAbortController = new AbortController();
      let entityIDToFetch;
      let dataSource;
      let dataSourceSchema;
      let result;
      const timeRangeFilters = selectors.timeRangeFilters(state.analyzer[id]);

      // Inform the state that we've made the request. Without this, the middleware will try to make the request again
      // immediately.
      api.dispatch((0, _action.appRequestedResolverData)({
        id,
        parameters: databaseParameters
      }));
      try {
        const matchingEntities = await dataAccessLayer.entities({
          _id: databaseParameters.databaseDocumentID,
          indices: databaseParameters.indices,
          signal: lastRequestAbortController.signal
        });
        if (matchingEntities.length < 1) {
          // If no entity_id could be found for the _id, bail out with a failure.
          api.dispatch((0, _action.serverFailedToReturnResolverData)({
            id,
            parameters: databaseParameters
          }));
          return;
        }
        ({
          id: entityIDToFetch,
          schema: dataSourceSchema,
          name: dataSource
        } = matchingEntities[0]);
        result = await dataAccessLayer.resolverTree({
          dataId: entityIDToFetch,
          schema: dataSourceSchema,
          timeRange: timeRangeFilters,
          indices: databaseParameters.indices,
          ancestors: (0, _resolver_tree.ancestorsRequestAmount)(dataSourceSchema),
          descendants: (0, _resolver_tree.descendantsRequestAmount)()
        });
        const resolverTree = {
          originID: entityIDToFetch,
          nodes: result
        };
        if (resolverTree.nodes.length === 0) {
          const unboundedTree = await dataAccessLayer.resolverTree({
            dataId: entityIDToFetch,
            schema: dataSourceSchema,
            indices: databaseParameters.indices,
            ancestors: (0, _resolver_tree.ancestorsRequestAmount)(dataSourceSchema),
            descendants: (0, _resolver_tree.descendantsRequestAmount)()
          });
          if (unboundedTree.length > 0) {
            const timestamps = unboundedTree.map(event => (0, _ecs_safety_helpers.firstNonNullValue)(event.data['@timestamp'])).sort();
            const oldestTimestamp = timestamps[0];
            const newestTimestamp = timestamps.slice(-1);
            api.dispatch((0, _action.serverReturnedResolverData)({
              id,
              result: {
                ...resolverTree,
                nodes: unboundedTree
              },
              dataSource,
              schema: dataSourceSchema,
              parameters: databaseParameters,
              detectedBounds: {
                from: String(oldestTimestamp),
                to: String(newestTimestamp)
              }
            }));

            // 0 results with unbounded query, fail as before
          } else {
            api.dispatch((0, _action.serverReturnedResolverData)({
              id,
              result: resolverTree,
              dataSource,
              schema: dataSourceSchema,
              parameters: databaseParameters
            }));
          }
        } else {
          api.dispatch((0, _action.serverReturnedResolverData)({
            id,
            result: resolverTree,
            dataSource,
            schema: dataSourceSchema,
            parameters: databaseParameters
          }));
        }
      } catch (error) {
        // https://developer.mozilla.org/en-US/docs/Web/API/DOMException#exception-AbortError
        if (error instanceof DOMException && error.name === 'AbortError') {
          api.dispatch((0, _action.appAbortedResolverDataRequest)({
            id,
            parameters: databaseParameters
          }));
        } else {
          api.dispatch((0, _action.serverFailedToReturnResolverData)({
            id,
            parameters: databaseParameters
          }));
        }
      }
    }
  };
}